<?php
/*******************************************************************************
 **                         Access2PostgreSQL Module                          **
 **                             September 6, 2007                             **
 **                                Version 2.0                                **
 **                                                                           **
 **                                  Author:                                  **
 **                               Nathan Bruer                                **
 **                                                                           **
 **                               Requirements:                               **
 **                                   PHP5                                    **
 **                           PostgreSQL Extension                            **
 **                              ODBC Extension                               **
 **                                                                           **
 **                                                                           **
 **                         Setup & Run instructions:                         **
 ** Configure the config.inc.php file located in the same directory as this   **
 **      file. Then run the index.php file in theconsole (command line).      **
 **                                                                           **
 **                              Problems/Bugs:                               **
 **             If you experiance problems, please contact me at:             **
 **              http://pgfoundry.org/projects/access2postgres/               **
 **                                                                           **
 **                                License:                                   **
 **   This software is licensed under the CC-GNU LGPL (Creative Commons GNU   **
 **    Lesser General Public License). You may find the full license here:    **
 **               http://creativecommons.org/licenses/LGPL/2.1/               **
 ******************************************************************************/
define('VERBOSE_MUST_SHOW', -2); // This is for special thins, like credits
define('VERBOSE_USER_INPUT', -1); // Will always display (message for user to respond to)
define('VERBOSE_CRITICAL', 0); // Will always display (fatal errors, [script will stop!])
define('VERBOSE_ERROR',1); // Should always display (error that can be fixed [?user input required?])
define('VERBOSE_WARNING', 2); // Should display (problems or couldn't do something, but can continue)
define('VERBOSE_MESSAGE', 3); // Can display (status)
define('VERBOSE_INFO', 4); // Might display (debugging, complete output)
/**
 * The master console class that handles all transactions at this point
 * @author Nathan Bruer
 * @package Access2PostgreSQL
 */
class Console {
	/**
	 * This is a temporary variable used to pass the key of the table.
	 * uses the following format: array('tablename1' => 'serial_column_name')
	 * @access private
	 */
	private $serialKey = array();
	/**
	 * This is an array of items describing the current status using the following format:
	 * array(0 => {row being processed}, 1 => {total number of rows}, 2 => {table name})
	 * @access private
	 */
	private $status;
	/**
	 * This variable is the timestamp of when the script was started
	 * (used to calculate how long it was running)
	 * @access private
	 */
	private $startTime;
	/**
	 * Construct function, this does most of the handling of what to do next
	 * @access public
	 */
	public function __construct(){
		global $config;
		$this->startTime = microtime(true);
		// Displays the credits (YOU MAY NOT REMOVE THIS!)
		self::showCredits();
		// Checks for a config variable and if not displays error
		if(!$config) self::vPrint('***You need to first setup the conf.php file***', VERBOSE_CRITICAL);
		self::vPrint("I am now going to try and connect to the Microsoft Access Database...\n", VERBOSE_MESSAGE);
		self::vPrint("I am now sending: '" . sprintf($config->Access->Driver, $config->Access->Location) . "' with username: \"{$config->Access->Username}\" and password: \"{$config->Access->Password}\" to the ODBC Driver\n", VERBOSE_INFO);
		// Tries to connect to the odbc database, uppon failure asks if you would like to retry
		while(!($config->Access->Link = @odbc_connect(sprintf($config->Access->Driver, $config->Access->Location), $config->Access->Username, $config->Access->Password))){
			self::vPrint("I could not connect to the Microsoft Access Database, because of the\nfollowing error:\n\n", VERBOSE_ERROR);
			self::vPrint("Code: " . odbc_error() . ", Message: " . odbc_errormsg() . "\n\n", VERBOSE_WARNING);
			while(true){
				// Displays error and asks for user input
				self::vPrint("What would you like todo? (r=retry,a=abort) ", VERBOSE_USER_INPUT);
				$tmpResponce = substr(trim(fgets(STDIN)), 0, 1);
				if(!strcasecmp($tmpResponce, 'a')){
					// Displays error and kills the script
					self::vPrint("Quit at users request!\n", VERBOSE_CRITICAL);
				}elseif(!strcasecmp($tmpResponce, 'r')){
					break 1;
				}else{
					self::vPrint("Invalid entry. Please specify again!\n", VERBOSE_ERROR);
					continue 1;
				}
			}
		}
		self::vPrint("Success! I am now connected to the Microsoft Access Database\n", VERBOSE_MESSAGE);
		self::vPrint("I am now going to try and connect to the PostgreSQL Database...\n", VERBOSE_MESSAGE);
		self::vPrint("I am now sending: \"host='{$config->PostgreSQL->Host}' port='{$config->PostgreSQL->Port}' dbname='{$config->PostgreSQL->Database}' user='{$config->PostgreSQL->Username}' password='{$config->PostgreSQL->Password}'\" to PostgreSQL\n", VERBOSE_INFO);
		// Tries to connect to the PostgreSQL database, uppon failure asks if you would like to retry
		while(!($config->PostgreSQL->Link = pg_connect("host='{$config->PostgreSQL->Host}' port='{$config->PostgreSQL->Port}' dbname='{$config->PostgreSQL->Database}' user='{$config->PostgreSQL->Username}' password='{$config->PostgreSQL->Password}'"))){
			self::vPrint("I could not connect to the PostgreSQL Database, because of the following error:\n", VERBOSE_ERROR);
			self::vPrint(pg_last_error() . "\n", VERBOSE_ERROR);
			// Displays error and asks for user input
			self::vPrint("What would you like todo? (r=retry,a=abort) ", VERBOSE_USER_INPUT);
			$tmpResponce = substr(trim(fgets(STDIN)), 0, 1);
			if(strcasecmp($tmpResponce, 'a')){
				self::vPrint("Quit at users request!\n", VERBOSE_CRITICAL);
			}elseif(strcasecmp($tmpResponce, 'r')){
				continue;
			}else{
				self::vPrint("Invalid entry. Please specify again!\n", VERBOSE_ERROR);
				continue;
			}
		}
		self::vPrint("Success! I am now connected to the PostgrSQL Database!\n", VERBOSE_MESSAGE);
		self::vPrint("I am now going to get the list of tables from your Access Database...\n", VERBOSE_MESSAGE);
		// Gets the list of tables from the Access database
		$config->Access->Tables = self::fetchAccessTables();
		self::vPrint("Successfully retrieved list of tables from the Access Database.\n", VERBOSE_MESSAGE);
		self::vPrint("I am now going to get the rows and data type for each table in Access\n", VERBOSE_MESSAGE);
		// Gets each row type and info about each row in each table and ports it to PostgreSQL
		$tables = self::configureTableInfo($config->Access->Tables);
		self::vPrint("I am now finished creating the PostgreSQL database table(s)\n", VERBOSE_MESSAGE);
		// Converts the actual data in the Access database
		self::convertTableData($tables);
		// Calculates time taken and displays it
		$timeTook = microtime(true) - $this->startTime;
		$days = floor($timeTook / 86400);
		$hours = floor(($timeTook = $timeTook - ($days * 86400)) / 3600);
		$mins = floor(($timeTook = $timeTook - ($hours * 3600)) / 60);
		$secs = ($timeTook - ($hours * 60));
		self::vPrint("Time it took to convert: {$days}d {$hours}h {$mins}m {$secs}s\n", VERBOSE_MESSAGE);
		self::vPrint("\n\nI am finished converting the database as best as I can. Since Access\ndatabases have no way to retrieve the current counter of the id, please open\nyour Access database and set each of the sequences of the tables in postgresQL\nas the sequences are in the Access database. NOTE: This step may not be\nnecessary.\n\n-Nathan Bruer (sole developer)\n", VERBOSE_MUST_SHOW);
		
	}
	/**
	 * Closes open connections
	 * @access public
	 */
	public function __destruct(){
		global $config;
		@odbc_close_all();
		@pg_close($config->PostgreSQL->Link);
	}
	/**
	 * Converss the actual data from a table
	 * @param array $tables A list of tables in the following format: array('tablename', 'tablename2')
	 * @access protected
	 */
	protected function convertTableData($tables){
		global $config;
		foreach($tables as $value){
			self::vPrint("I am now going to convert the data from the \"$value\" table...\n", VERBOSE_MESSAGE);
			// Prepares and executes the SQL query to Access to fetch all data from the current table
			$accessSQL = "SELECT * FROM $value";
			$accessResult = self::accessQuery($accessSQL);
			self::vPrint("I have called for the data and am going to extract and convert it...\n", VERBOSE_MESSAGE);
			// Prepares and executes the SQL query to PostgreSQL to send all the data via stdin
			$pgSQL = "COPY \"$value\" FROM stdin";
			self::vPrint("I am going to open a STDIN to postgres to input the data using the following command: $pgSQL", VERBOSE_INFO);
			self::pgQuery($pgSQL);
			
			// Calculates the number of rows in a table for status purpises. Upon failure it does it using a SQL query
			if(($this->status[1] = odbc_num_rows($accessResult)) === -1){
				self::vPrint("I am now going to find out how many rows are in the Access table $value...\n", VERBOSE_INFO);
				$row = odbc_fetch_array(self::accessQuery("SELECT COUNT(*) as count FROM $value"));
				$this->status[1] = $row['count'];
			}
			// Sets the status to the current table
			$this->status[2] = $value;

			for($i=1;$row = odbc_fetch_array($accessResult,$i);$i++){
				// If the tick hits the number of ticks it displays the status
				if($i%$config->Settings->Ticks === 0){
					// Sets the status to the current row being processed
					$this->status[0] = $i;
					self::status_check();
				}
				// If the column is null sets it to \N and replaces special string types with corresponding data
				foreach($row as $fieldName => &$col)
					if($col === null)
						$col = "\\N";
					elseif(is_string($col))
						$col = str_replace(array("\0", "\r", "\n", "\t", "\\", "\x08", "\x0C", "\x0B"), array("\\x00", "\\r", "\\n", "\\t", "\x5C", "\\b", "\\f", "\\v"), $col);
				// Joins all the tables together by a tab followed by a new line character
				$pgLineInput = implode("\t", $row) . "\n";
				self::vPrint("I am now going to send PostgrSQL the following:\n$pgLineInput", VERBOSE_INFO);
				// Sends the line to PostgreSQL
				pg_put_line($config->PostgreSQL->Link, $pgLineInput);
			}
			// Tells PostgreSQL that we are done
			pg_put_line($config->PostgreSQL->Link, "\\.\n");
			// Ends the stdin to PostgreSQL
			if(!pg_end_copy($config->PostgreSQL->Link)){
				self::vPrint("There was an error with the input of all the data, however I can continue. Table: \"$value\"\n", VERBOSE_ERROR);
			}
			self::vPrint("I am now going to set the serial field to what the highest value...\n", VERBOSE_MESSAGE);
			// The serial field is the key field. This estimates what it would be.
			// Since Access databases have now current way to retrieve the current key state this
			// goes off of what the highest value is (message is given at the end)
			if(@$this->serialKey[$value] !== null){
				$pgSQL = "SELECT MAX(\"{$this->serialKey[$value]}\") AS \"max\" FROM \"$value\"";
				self::vPrint("I am now sending: $pgSQL\n", VERBOSE_INFO);
				$result = self::pgQuery($pgSQL);
				$row = pg_fetch_assoc($result);
				if(((int) $row['max']) <= 0) $row['max'] = 1;
				$pgSQL = "ALTER SEQUENCE \"{$value}_{$this->serialKey[$value]}_seq\" RESTART WITH $row[max]";
				self::vPrint("I am now sending: $pgSQL", VERBOSE_INFO);
				self::pgQuery($pgSQL);
				self::vPrint("I have set the serial key to $row[max]\n", VERBOSE_MESSAGE);
			}
			self::vPrint("I am now finished converting table: \"$value\"\n", VERBOSE_MESSAGE);
		}
		self::vPrint("I am now finished converting all the tables\n", VERBOSE_MESSAGE);
	}
	/**
	 * Handles a status check
	 * @param boolean $return weather to return the status vs echo it
	 * @return string Status as a string (if applicable)
	 */
	public function status_check($return = false){
		if($return) return "Status Check: Processing row {$this->status[0]} of {$this->status[1]} in table {$this->status[2]}...";
		self::vPrint("Status Check: Processing row {$this->status[0]} of {$this->status[1]} in table {$this->status[2]}...\n", VERBOSE_MESSAGE);
	}
	/**
	 * Reads the table names from $tableData and minipulates it to the column info.
	 * Also adds the new tables to PostgreSQL
	 * @param array $tableData  Variable that has the list of table names.
	 * @return array List of table names
	 */
	protected function configureTableInfo($tableData){
		global $config;
		$tables = array();
		foreach($tableData as $tableName => &$tableInfo){
			$tables[] = $tableName;
			self::vPrint("Retrieving column list for table: $tableName in Access\n", VERBOSE_INFO);
			// Gets the list of columns from the ODBC driver
			$colSQL = odbc_columns($config->Access->Link, $config->Access->DBName, null, $tableName);
			while($columnInfo = odbc_fetch_array($colSQL)){
				// Handles the ODBC type conversion to PostgreSQL type
				if(($pgType = self::getAccess2PGTypeConversion($columnInfo['TYPE_NAME'], $columnInfo['COLUMN_SIZE'])) !== false){
					$tableInfo[] = "\"$columnInfo[COLUMN_NAME]\" $pgType";
					if($pgType === "serial"){
						$this->serialKey[$tableName] = $columnInfo['COLUMN_NAME'];
					}
				}else{
					self::vPrint("I was unable to figure out what data type this is: {$columnInfo['TYPE_NAME']} in table $tableName in column $columnInfo[COLUMN_NAME]\n", VERBOSE_CRITICAL);
				}
			}
			// Checks to see if it's a serial type and if so sets it
			if(!isset($this->serialKey[$tableName])){
				$this->serialKey[$tableName] = null;
			}
			// Sends the sql query to create the table.
			$createTableSQL = "CREATE TABLE \"$tableName\"\n(\n\t" . implode(",\n\t", $tableInfo) . "\n) WITHOUT OIDS;";
			self::vPrint("I am now going to create table \"$tableName\" in PostgreSQL\n", VERBOSE_MESSAGE);
			self::vPrint("I am sending PostgreSQL this SQL: $createTableSQL\n", VERBOSE_INFO);
			self::pgQuery($createTableSQL);
		}
		return $tables;
	}
	/**
	 * Handles the Access query opperations
	 * @access protected
	 * @return refrence The Result set of the passed query
	 */
	protected function accessQuery($SQL){
		global $config;
		if(($result = @odbc_exec($config->Access->Link, $SQL)) === false){
			self::vPrint("There was a SQL error with the following Access SQL:\n$SQL\nThe error was:\n".odbc_errormsg($config->Access->Link), VERBOSE_CRITICAL);
		}
		return $result;
	}
	/**
	 * Handles the PostgreSQL query opperations
	 * @access protected
	 * @return refrence The Result set of the passed query
	 */
	protected function pgQuery($SQL){
		global $config;
		if(($result = @pg_query($config->PostgreSQL->Link, $SQL)) === false){
			self::vPrint("There was a SQL error with the following PostgreSQL SQL:\n$SQL\nThe message was:\n".pg_last_error($config->PostgreSQL->Link), VERBOSE_CRITICAL);
		}
		return $result;
	}
	/**
	 * Used to return the data type conversion from Access to PostgreSQL
	 * @param string $type A string containing the type to be converted
	 * @param mixed $size Usually an numeric value containing the size of the field
	 * @return string|bool Returns either false if it couldnt figure out or the type conversion
	 */
	protected function getAccess2PGTypeConversion($type, $size){
		$type = strtolower($type);
		/* I found this link very usefull:  http://office.microsoft.com/en-us/access/HP052745731033.aspx */
		switch($type){
			case 'counter':
				return 'serial';
			case 'double':
				return 'double precision';
			case 'currency':
				return 'numeric';
			case 'numeric':
				return 'numeric';
			case 'single':
				return 'real';
			case 'hyperlink':
				return "varchar($size)";
			case 'longchar':
				return 'text';
			case 'bit':
				return 'bool';
			case 'byte':
				return 'numeric';
			case 'tinyint':
				return 'smallint';
			case 'smallint';
				return 'smallint';
			case 'integer':
				return 'integer';
			case 'int':
				return 'integer';
			case 'real':
				return 'real';
			case 'bigint':
				return 'bigint';
			case 'float':
				return 'double precision';
			case 'money':
				return 'numeric';
			case 'smallmoney':
				return 'numeric';
			case 'decimal':
				return 'decimal';
			case 'numeric':
				return 'numeric';
			case 'datetime':
				return 'timestamp without time zone';
			case 'smalldatetime':
				return 'timestamp without time zone';
			case 'varchar':
				return "varchar($size)";
			/*
				nvarchar(n) (nvarchar(n) data type: In an Access project, a
				variable-length data type with a maximum of 4,000 Unicode
				characters. Unicode characters use 2 bytes per character and
				support all international characters.)
			*/
			case 'nvarchar':
				return 'text'; // NOT SURE ABOUT THIS ONE
			case 'text':
				return 'text';
			case 'image':
				return 'bytea';
			case 'longbinary':
				return 'bytea';
			/*
				uniqueidentifier (uniqueidentifier data type: In an Access
				project, a 16-byte globally unique identifier (GUID).) (SQL
				Server 7.0 or later)
			*/
			case 'uniqueidentifier':
				return false; // NOT SURE???
			case 'char':
				return 'text';
			case 'nchar':
				return 'text';
			case 'varbinary':
				return 'bytea';
			case 'smallint':
				return 'smallint';
			/*
				timestamp (timestamp data type: In an Access project, a data
				type that is automatically updated every time a row is
				inserted or updated. Values in timestamp columns are not
				datetime data, but binary(8) or varbinary(8), indicating the
				sequence of data modifications.)
			*/
			case 'timestamp':
				return false; // NOT SURE?
			case 'char':
				return "varchar($size)";
			case 'nchar':
				return "varchar($size)";
			/*
				sql_variant (sql_variant data type: In an Access project, a data
				type that stores values of several data types, except for text,
				ntext, image, timestamp, and sql_variant types. It is used in a
				column, parameter, variable, or return value of a user-defined
				function.)
			*/
			case 'sql_variant':
				return false; // NOT SURE?
			/*
				user-defined (user-defined data type: In a Microsoft SQL Server
				database, a definition of the type of data a column can contain. It
				is defined by the user with existing system data types. Rules and
				defaults can only be bound to user-defined data types.)
			*/
			case 'user-defined':
				return false; // NOT SURE? OID?
			default:
				return false;
		}
	}
	/**
	 * Gets a list of tables from the access database as an assoc array.
	 * Array format:
	 * %return% = array("Table Name 1" = array(), "Table Name 2" = array())
	 *
	 * @return array A list of tables in assoc format
	 */
	protected function fetchAccessTables(){
		global $config;
		$tablelist = array();
		// Fetches the tables list
		$query = odbc_tables($config->Access->Link);
		while($table=odbc_fetch_array($query)){
			// This script needs to know the name of the Access database this retrieves it
			if(!isset($config->Access->DBName)) $config->Access->DBName = $table['TABLE_CAT'];
			// Checks to see if it's a table
			if($table['TABLE_TYPE'] === "TABLE"){
				self::vPrint("Table found: {$table['TABLE_NAME']}\n", VERBOSE_INFO);
				$tablelist[$table['TABLE_NAME']] = array();
			}
		}
		/* The retrieves the database name */
		return $tablelist;
	}
	/**
	 * This function is used to return data to the user.
	 * @param string $message The message to send to the user
	 * @param integer $errorNum The importance of the message (ie. VERBOSE_ERROR, VERBOSE_MESSAGE, etc...)
	 */
	protected function vPrint($message, $errorNum){
		global $config;
		if(!isset($config->Settings->LogLevel)){
			$config->Settings->LogLevel = 3;
		}
		// If it's crtitical it will show and quit the script.
		if($errorNum === VERBOSE_CRITICAL){
			echo $message;
			exit;
		// If it's less than 0 or less than or equal to the loglevel setting it shows it.
		}elseif(($errorNum < 0) || ($errorNum <= $config->Settings->LogLevel)){
			echo $message;
		}
	}
	/**
	 * This function shows credits.
	 */
	private function showCredits(){
		self::vPrint(
				"********************************************************************************\r".
				"**                              Access2PostgreSQL                             **\r".
				"**                              September 6, 2007                             **\r".
				"**                                 Version 1.1                                **\r".
				"**                                                                            **\r".
				"**                                   Author:                                  **\r".
				"**                                Nathan Bruer                                **\r".
				"**                                                                            **\r".
				"**                               Problems/Bugs:                               **\r".
				"**              If you experiance problems, please contact me at:             **\r".
				"**               http://pgfoundry.org/projects/access2postgres/               **\r".
				"**                                                                            **\r".
				"**    This software is licensed under the CC-GNU LGPL (Creative Commons GNU   **\r".
				"**     Lesser General Public License). You may find the full license here:    **\r".
				"**                http://creativecommons.org/licenses/LGPL/2.1/               **\r".
				"********************************************************************************\r\n",
				VERBOSE_MUST_SHOW);
	}
}
?>