﻿using System;
using System.Data;
using System.Data.Common;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Text;

namespace fb2pg
{
    /// <summary>
    /// Esta clase gestiona una clave primaria de una tabla. Básicamente almacena el nombre de la tabla y el de los campos que componen la clave primaria.
    /// Utilizamos esta clase para crear una lista con los nombres de las tablas y sus claves primarias
    /// </summary>
    class TablasClavesPrimarias
    {
        #region · Campos ·

        /// <summary>
        ///  Nombre de la tabla
        /// </summary>
        string tabla;
        /// <summary>
        ///  Nombre de la tabla
        /// </summary>
        public string Tabla
        {
            get { return tabla; }
            set { tabla = value; }
        }

        /// <summary>
        /// Nombres de los campos, separados por comas, que componen la clave primaria
        /// </summary>
        string campos;
        /// <summary>
        /// Nombres de los campos, separados por comas, que componen la clave primaria
        /// </summary>
        public string Campos
        {
            get { return campos; }
            set { campos = value; }
        }

        #endregion

        #region · Constructor ·

        /// <summary>
        /// Crea una instancia de la clase <see cref="TablasClavesPrimarias"/>
        /// </summary>
        /// <param name="t">Nombre de la tabla</param>
        /// <param name="c">Nombres de los campos, separados por comas, que componen la clave primaria</param>
        public TablasClavesPrimarias(string t, string c)
        {
            this.tabla = t.Trim().ToLowerInvariant();
            this.campos = c.Trim().ToLowerInvariant();
        }

        #endregion

        #region · Métodos estáticos ·

        /// <summary>
        /// Rellena una lista con las tablas y sus claves primarias asociadas que contiene la base de datos Firebird
        /// </summary>
        /// <param name="conn">Conexión con el servidor Firebird</param>
        /// <param name="lista">Lista de objetos <see cref="TablasClavesPrimarias"/> que vamos a rellenar</param>
        public static void RellenaListaClavesPrimarias(DbConnection conn, List<TablasClavesPrimarias> lista)
        {
            if (conn == null || lista == null)
                return;
            
            using (DbCommand cmd = conn.CreateCommand())
            {
                NameValueCollection valores = new NameValueCollection();
                cmd.CommandText =
                    "SELECT RDB$RELATION_CONSTRAINTS.RDB$RELATION_NAME, RDB$INDEX_SEGMENTS.RDB$FIELD_NAME " +
                    "FROM RDB$RELATION_CONSTRAINTS, RDB$INDEX_SEGMENTS " +
                    "WHERE " +
                    "RDB$RELATION_CONSTRAINTS.RDB$CONSTRAINT_TYPE = 'PRIMARY KEY' AND " +
                    "RDB$INDEX_SEGMENTS.RDB$INDEX_NAME=RDB$RELATION_CONSTRAINTS.RDB$INDEX_NAME " +
                    "ORDER BY RDB$RELATION_CONSTRAINTS.RDB$RELATION_NAME, RDB$INDEX_SEGMENTS.RDB$FIELD_POSITION ";
                using (DbDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        valores.Add(dr.GetString(0).Trim().ToLowerInvariant(), dr.GetString(1).Trim().ToLowerInvariant());
                    }
                    dr.Close();
                }

                for (int i = 0; i < valores.Count; i++)
                {
                    lista.Add(new TablasClavesPrimarias(valores.GetKey(i), valores.Get(i)));
                }
            }
        }

        /// <summary>
        /// Devuelve los campos que componen la clave primaria de una tabla
        /// </summary>
        /// <param name="lista">Lista de objetos <see cref="TablasClavesPrimarias"/> en la que buscamos</param>
        /// <param name="nombreTabla">Nombre de la tabla que buscamos</param>
        /// <returns>
        /// Si la tabla está en la lista, devuelve los campos que componen su clave primaria.
        /// En caso contrario, devuelve una cadena de texto vacía
        /// </returns>
        public static string BuscaCamposTabla(List<TablasClavesPrimarias> lista, string nombreTabla)
        {
            String campos = String.Empty;

            TablasClavesPrimarias item = lista.Find(delegate(TablasClavesPrimarias t) { return t.tabla.Equals(nombreTabla); });
            if (item != null)
                campos = item.campos;

            return campos;
        }

        #endregion
    }
}
