/* $Id: type.c,v 1.12 2005/06/28 00:27:38 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/type.c,v 1.7 2004/12/15 05:15:27 flaw
 * if/src/type.c,v 1.11 2004/09/28 15:48:01 flaw
 *//*
 * Postgres type interface
 */
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_cast.h>
#include <nodes/params.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/tupledesc.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/utils.h>
#include <pypg/error.h>
#include <pypg/conv.h>
#include <pypg/tif.h>

PyObj
type_cast(PyObj self, PyObj ob)
{
	HeapTuple tup;
	Oid from, to, castor;
	volatile Datum rd = 0;
	volatile PyObj rob = NULL;

	if (!PyPgObject_Check(ob))
	{
		PyErr_Format(PyExc_TypeError,
			"expecting a Postgres.Object, got %s",
			ob->ob_type->tp_name);
		return(NULL);
	}
	
	from = ObjectIdGetDatum(PyPgObject_FetchTypeOid(ob));
	to = ObjectIdGetDatum(PyPgObject_FetchOid(self));
	tup = SearchSysCache(CASTSOURCETARGET, from, to, 0, 0);
	if (tup == NULL)
	{
		char *fromTypeStr, *toTypeStr;
		fromTypeStr = PgTypeName_FromOid(from);
		toTypeStr = PgTypeName_FromOid(to);

		PyErr_Format(PyExc_TypeError,
				"no function for casting a '%s' object to a '%s' object",
				fromTypeStr, toTypeStr);

		pfree(fromTypeStr);
		pfree(toTypeStr);
		return(NULL);
	}

	castor = CASTSTRUCT(tup)->castfunc;
	ReleaseSysCache(tup);

	PgError_TRAP(rd = OidFunctionCall1(castor, PyPgObject_FetchDatum(ob)));
	if (!PyErr_Occurred())
		rob = PyPgObject_New((PyObj)self, rd);

	return(rob);
}

PyObj
type_binary(PyObj self, PyObj bin)
{
	RETURN_NONE;
}

PyObj
type_string(PyObj self, PyObj bin)
{
	RETURN_NONE;
}

static PyMethodDef PyPgType_Methods[] = {
	{"cast", (PyCFunction) type_cast, METH_O,
	"cast a Postgres.Object to a new object of this type"},
	{"binary", (PyCFunction) type_binary, METH_O,
	"create a Postgres.Object of this type from the given raw data "
	"and this type's binary input function"},
	{"string", (PyCFunction) type_string, METH_O,
	"create a Postgres.Object of this type from the given string "
	"and this type's text input function"},
	{NULL}
};

static PyMemberDef PyPgType_Members[] = {
	{"TupleDesc", T_OBJECT, offsetof(struct PyPgType, pgt_desc), RO,
	"Postgres.Type's corresponding Relation's Postgres.TupleDesc or "
	"a single column "},
	{"NULL", T_OBJECT, offsetof(struct PyPgType, pgt_null), RO,
	"Postgres.Type's NULL object, None if the pg_type entry's typnotnull "
	"is true"},
	{NULL}
};

static int
type_traverse(PyObj self, visitproc visit, void *arg)
{
	int err = 0;
	PyObj ob;

	ob = PyPgType_FetchPyPgTupleDesc(self);
	if (ob != NULL)
	{
		err = visit(ob, arg);
		if (err) return(err);
	}

	ob = PyPgType_FetchNULL(self);
	if (ob != NULL && ob != Py_None)
	{
		err = visit(ob, arg);
		if (err) return(err);
	}

	err = self->ob_type->tp_base->tp_traverse(self, visit, arg);
	return(err);
}

static int
type_clear(PyObj self)
{
	int r = 0;
	PyObj ob;

	ob = PyPgType_FetchNULL(self);
	if (ob != NULL)
	{
		PyPgType_FixNULL(self, NULL);
		DECREF(ob);
	}

	ob = PyPgType_FetchPyPgTupleDesc(self);
	if (ob != NULL)
	{
		PyPgType_FixPyPgTupleDesc(self, NULL);
		DECREF(ob);
	}

	r = self->ob_type->tp_base->tp_clear(self);
	return(r);
}

static PyObj
type_repr(PyObj self)
{
	PyObj rob;
	rob = PyString_FromString(PyPgType_FetchName(self));
	return(rob);
}

static PyObj
type_call(PyObj self, PyObj args, PyObj kw)
{
	volatile PyObj ob = NULL, rob = NULL;
	Form_pg_type ts;

	if (PyPgObject_Check(args) && PyPgObject_FetchType(args) == self)
		return(args);

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(NULL);

	ts = PyPgType_FetchTypeStruct(self);
	
	if (ts->typrelid)
	{
		TupleDesc td;
		HeapTuple ht;
		td = PyPgType_FetchTupleDesc(self);
		ht = HeapTuple_FromTupleDescAndIterable(td, args);
		if (ht == NULL)
			return(NULL);
		rob = PyPgObject_FromPyPgTypeAndHeapTuple(self, ht);
		heap_freetuple(ht);
	}
	else if (ts->typelem != 0 && ts->typlen == -1)
	{
		ArrayType *ar;
		Oid at = PyPgType_FetchTypeStruct(self)->typelem;
		ar = Array_FromTypeOidAndPySequence(at, args);
		rob = PyPgObject_New(self, PointerGetDatum(ar));
	}
	else
	{
		if (!PyArg_ParseTuple(args, "O", &ob))
			return(NULL);
			
		if (PyPgObject_Check(ob))
		{
			if (PyPgObject_FetchTypeOid(ob) == PyPgObject_FetchOid(self))
			{
				INCREF(ob);
				rob = ob;
			}
			else
				rob = type_cast(self, ob);
#if 0
			HeapTuple tup;
			Oid from, to, castor;
			volatile Datum rd = 0;

			from = ObjectIdGetDatum(PyPgObject_FetchTypeOid(ob));
			to = ObjectIdGetDatum(PyPgObject_FetchOid(self));
			tup = SearchSysCache(CASTSOURCETARGET, from, to, 0, 0);
			if (tup == NULL)
			{
				char *fromTypeStr, *toTypeStr;
				fromTypeStr = PyPgTypeName_FromOid(from);
				toTypeStr = PyPgTypeName_FromOid(to);

				PyErr_Format(PyExc_TypeError,
						"no function for casting a '%s' object to a '%s' object",
						fromTypeStr, toTypeStr);

				pfree(fromTypeStr);
				pfree(toTypeStr);
				return(NULL);
			}

			castor = CASTSTRUCT(tup)->castfunc;
			ReleaseSysCache(tup);

			PgError_TRAP(rd = OidFunctionCall1(castor, PyPgObject_FetchDatum(ob)));
			if (!PyErr_Occurred())
				rob = PyPgObject_New(self, rd);
#endif
		}
		else if (ob == Py_None)
		{
			rob = PyPgType_FetchNULL(self);
			if (rob == NULL)
				rob = Py_None;

			INCREF(rob);
		}
		else
		{
			PyObj obstr;
			volatile Datum rd = 0;
			obstr = PyObject_Str(ob);
			if (obstr == NULL)
				return(NULL);

			switch (PyPgObject_FetchOid(self))
			{
				case BYTEAOID:
				{
					Size sz = PyString_GET_SIZE(obstr) + VARHDRSZ;
					PgError_TRAP(rd = PointerGetDatum(palloc(sz)));
					if (!PyErr_Occurred())
					{
						VARATT_SIZEP(rd) = sz;
						memcpy(
							VARDATA(rd),
							PyString_AS_STRING(obstr),
							PyString_GET_SIZE(obstr)
						);
					}
				}
				break;

				default:
					PgError_TRAP(
						rd = OidFunctionCall1(ts->typinput,
								PointerGetDatum(PyString_AS_STRING(obstr)))
					);
				break;
			}
			Py_DECREF(obstr);
			if (!PyErr_Occurred())
			{
				rob = PyPgObject_New(self, rd);
				if (rob == NULL && !(PyPgType_FetchTypeStruct(self)->typbyval))
					pfree(DatumGetPointer(rd));
			}
		}
	}

	return(rob);
}

static PyObj
type_new_from_oid(PyTypeObject *subtype, Oid typoid)
{
	Form_pg_type ts;
	HeapTupleHeader hth;
	HeapTuple ht;
	TupleDesc td;
	PyObj lo, tdo, rob;

	lo = PyLong_FromUnsignedLong(typoid);
	if (PyMapping_HasKey(PyPgType_Cache, lo))
	{
		rob = PyObject_GetItem(PyPgType_Cache, lo);
		Py_DECREF(lo);
		return(rob);
	}

	ht = SearchSysCache(TYPEOID, ObjectIdGetDatum(typoid), 0, 0, 0);
	if (ht == NULL)
	{
		PyErr_Format(PyExc_LookupError,
			"failed to lookup pg_type entry for type whose Oid = %d", typoid);
		goto free_lo;
	}

	hth = HeapTupleHeader_FromHeapTuple(ht);
	ReleaseSysCache(ht);

	if (hth == NULL)
	{
		PyErr_Format(PyExc_MemoryError,
			"failed to allocate memory for type's(%u) Datum", typoid);
		goto free_lo;
	}

	HeapTupleHeaderSetOid(hth, typoid);
	ts = (Form_pg_type) HeapTupleHeaderStructure(hth);

	if (ts->typrelid != InvalidOid)
	{
		tdo = PyPgTupleDesc_FromRelationId(ts->typrelid);
	}
	else
	{
		/* XXX: trap error */
		td = CreateTemplateTupleDesc(1, false);
		if (td == NULL)
			goto free_hth;
		td->tdtypeid = typoid;
		TupleDescInitEntry(td, 1, ":generated:", typoid, -1, 0, false);
		tdo = PyPgTupleDesc_New(td);
	}

	if (tdo == NULL)
		goto free_hth;

	rob = subtype->tp_alloc(subtype, 0);
	if (rob == NULL)
		goto free_tdo;

	PyPgObject_FixDatum(rob, PointerGetDatum(hth));
	PyPgObject_FixType(rob, NULL);

	PyPgType_FixPyPgTupleDesc(rob, tdo);
	PyPgType_FixNULL(rob, Py_None);

	if (ts->typnotnull == false)
	{
		PyObj nob = PyPgObject_NEW();
		if (nob == NULL)
			goto free_rob;
		PyPgObject_FixDatum(nob, 0);
		PyPgObject_FixType(nob, rob);
		PyPgType_FixNULL(rob, nob);
		Py_INCREF(rob);
	}
	else
		Py_INCREF(Py_None);

	PyObject_SetItem(PyPgType_Cache, lo, rob);

	if (HeapTupleHeaderGetTypeId(hth) == typoid)
	{
		PyPgObject_FixType(rob, rob);
	}
	else
	{
		PyObj rt = type_new_from_oid(subtype, HeapTupleHeaderGetTypeId(hth));
		if (rt == NULL)
			goto free_tdo;
		PyPgObject_FixType(rob, rt);
	}

	INCREF(PyPgObject_FetchType(rob));

	return(rob);

free_rob:
	rob->ob_type->tp_free(rob);
free_tdo:
	Py_DECREF(tdo);
free_hth:
	pfree(hth);
free_lo:
	Py_DECREF(lo);

	return(NULL);
}

PyObj
type_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	volatile PyObj ob = NULL, rob = NULL;
	volatile Oid typoid = InvalidOid;

	if (PyCFunctionErr_NoKeywordsAllowed(kw)
		|| !PyArg_ParseTuple(args, "O", &ob))
		return(NULL);

	PgError_TRAP(typoid = TypeOid_FromPyObject(ob));
	if (PyErr_Occurred())
		return(NULL);

	if (typoid == InvalidOid)
	{
		if (!PyErr_Occurred())
		{
			PyObj obstr;
			obstr = PyObject_Str(ob);
			PyErr_Format(PyExc_TypeError,
				"Postgres Type '%s' does not exist",
				PyString_AS_STRING(obstr));
			DECREF(obstr);
		}
	}
	else
	{
		rob = type_new_from_oid(subtype, typoid);
	}

	return(rob);
}

static char PyPgTypeType_Doc[] = 
"Python interface for the Postgres type";

PyTypeObject PyPgType_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Type",					/* tp_name */
	sizeof(struct PyPgType),		/* tp_basicsize */
	0,										/* tp_itemsize */
	NULL,									/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	type_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	type_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,				/* tp_flags */
	(char *) PyPgTypeType_Doc,		/* tp_doc */
	type_traverse,						/* tp_traverse */
	type_clear,							/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgType_Methods,					/* tp_methods */
	PyPgType_Members,					/* tp_members */
	NULL,									/* tp_getset */
	&PyPgObject_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	type_new,							/* tp_new */
};

PyObj
PyPgType_FromOid(Oid to)
{
	return(type_new_from_oid(&PyPgType_Type, to));
}

PyObj
PyPgType_FromRelationOid(Oid rid)
{
	Oid rto;
	rto = RelationId_FetchTypeOid(rid);
	return(PyPgType_FromOid(rto));
}

PyObj
PyPgType_FromPyObject(PyObj ob)
{
	volatile Oid typoid = InvalidOid;
	volatile PyObj rob = NULL;

	/* XXX: need to look into getting this changed */
	PgError_TRAP(typoid = TypeOid_FromPyObject(ob));
	if (typoid == InvalidOid)
	{
		if (!PyErr_Occurred())
		{
			PyObj obstr;
			obstr = PyObject_Str(ob);
			PyErr_Format(PyExc_TypeError,
				"Postgres Type '%s' does not exist", PYASSTR(obstr));
			DECREF(obstr);
		}
	}
	else
		rob = type_new_from_oid(&PyPgType_Type, typoid);

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
