/* $Id: geometric.c,v 1.11 2005/12/29 18:43:02 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Interface to geometric types
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_namespace.h>
#include <catalog/pg_conversion.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_opclass.h>
#include <catalog/namespace.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/geo_decls.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/relcache.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/numeric.h>
#include <pypg/type/geometric.h>
#include <pypg/type/textual.h>
#include <pypg/error.h>

#ifdef PGTYPE_POINT
PyObj
PyPg_point_FromPySequence(PyTypeObject *subtype, PyObj source)
{
	Point P;
	PyObj x, y, sx, sy, rob;

	if (PySequence_Length(source) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"point requires a two-length sequence, given %d",
			PySequence_Length(source));
		return(NULL);
	}

	sx = PySequence_GetItem(source, 0);
	if (sx == NULL) return(NULL);
	sy = PySequence_GetItem(source, 1);
	if (sy == NULL) return(NULL);

	x = PyNumber_Float(sx);
	Py_DECREF(sx);
	if (x == NULL) return(NULL);

	y = PyNumber_Float(sy);
	Py_DECREF(sy);
	if (y == NULL) return(NULL);

	P.x = PyFloat_AsDouble(x);
	Py_DECREF(x);
	P.y = PyFloat_AsDouble(y);
	Py_DECREF(y);

	rob = PyPgObject_New(subtype, PointPGetDatum(&P));
	return(rob);
}

PyObj
PyPg_point_FromPyObject(PyTypeObject *subtype, PyObj source)
{
	PyObj rob;

	if (PySequence_Check(source)
		&& !PyString_Check(source)
		&& !PyPg_text_CheckExact(source)
		&& !PyPg_varchar_CheckExact(source)
		&& !PyPg_cstring_CheckExact(source))
	{
		rob = PyPg_point_FromPySequence(subtype, source);
	}
	else
	{
		rob = PyPgObject_typinput((PyObj) subtype, source);
	}

	return(rob);
}

static int
point_length(PyObj self)
{
	return(2);
}

static PyObj
point_item(PyObj self, int item)
{
	Point *p = DatumGetPointP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(p != NULL);

	if (item > 1)
	{
		PyErr_Format(PyExc_IndexError,
			"index %d is out of range(2)", item);
		return(NULL);
	}

	rob = PyFloat_FromDouble(item == 0 ? p->x : p->y);
	return(rob);
}

static PySequenceMethods point_as_sequence = {
	point_length,		/* sq_length */
	NULL,					/* sq_concat */
	NULL,					/* sq_repeat */
	point_item,			/* sq_item */
	NULL,					/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	NULL,					/* sq_contains */
	NULL,					/* sq_inplace_concat */
	NULL,					/* sq_inplace_repeat */
};

#define PyPg_bool_returning_point_Method(fname, pg_func) \
static PyObj fname(PyObj self, PyObj ob) \
{ \
	Datum rd = 0; \
	PyObj rob; \
\
	ob = PyPg_point_FromPyObject(self->ob_type, ob); \
	if (ob == NULL) return(NULL); \
\
	PgError_TRAP( \
		rd = DirectFunctionCall2(pg_func, \
			PyPgObject_FetchDatum(self), \
			PyPgObject_FetchDatum(ob) \
		) \
	); \
	Py_DECREF(ob); \
	if (PyErr_Occurred()) return(NULL); \
\
	rob = rd == 0 ? Py_False : Py_True; \
	Py_INCREF(rob); \
\
	return(rob); \
}

#define PyPg_double_returning_point_Method(fname, pg_func) \
static PyObj fname(PyObj self, PyObj ob) \
{ \
	Datum rd = 0; \
	PyObj rob; \
\
	ob = PyPg_point_FromPyObject(self->ob_type, ob); \
	if (ob == NULL) return(NULL); \
\
	PgError_TRAP( \
		rd = DirectFunctionCall2(pg_func, \
			PyPgObject_FetchDatum(self), \
			PyPgObject_FetchDatum(ob) \
		) \
	); \
	Py_DECREF(ob); \
	if (PyErr_Occurred()) return(NULL); \
\
	rob = PyPgObject_New(&PyPg_float8_Type, rd); \
	pfree(DatumGetPointer(rd)); \
\
	return(rob); \
}

PyPg_bool_returning_point_Method(py_point_left, point_left);
PyPg_bool_returning_point_Method(py_point_right, point_right);
PyPg_bool_returning_point_Method(py_point_above, point_above);
PyPg_bool_returning_point_Method(py_point_below, point_below);
PyPg_bool_returning_point_Method(py_point_vertical, point_vert);
PyPg_bool_returning_point_Method(py_point_horizontal, point_horiz);
PyPg_double_returning_point_Method(py_point_slope, point_slope);
PyPg_double_returning_point_Method(py_point_distance, point_distance);

static PyMethodDef point_methods[] = {
	{"left", (PyCFunction) py_point_left, METH_O,
	PyDoc_STR("Whether the point is left of the given point")},
	{"right", (PyCFunction) py_point_right, METH_O,
	PyDoc_STR("Whether the point is right of the given point")},
	{"above", (PyCFunction) py_point_above, METH_O,
	PyDoc_STR("Whether the point is above the given point")},
	{"below", (PyCFunction) py_point_below, METH_O,
	PyDoc_STR("Whether the point is below the given point")},
	{"vertical", (PyCFunction) py_point_vertical, METH_O,
	PyDoc_STR("Whether the point is vertical with the given point")},
	{"horizontal", (PyCFunction) py_point_horizontal, METH_O,
	PyDoc_STR("Whether the point horizontal with the given point")},
	{"slope", (PyCFunction) py_point_slope, METH_O,
	PyDoc_STR("Calculate the slope of two points")},
	{"distance", (PyCFunction) py_point_distance, METH_O,
	PyDoc_STR("Calculate the distance between the two points")},
	{NULL}
};

static PyObj
point_get_x(PyObj self, void *closure)
{
	Point *p = DatumGetPointP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(p != NULL);

	rob = PyFloat_FromDouble(p->x);
	return(rob);
}

static PyObj
point_get_y(PyObj self, void *closure)
{
	Point *p = DatumGetPointP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(p != NULL);

	rob = PyFloat_FromDouble(p->y);
	return(rob);
}

static PyGetSetDef point_getset[] = {
	{"x", point_get_x, NULL, PyDoc_STR("The horizontal magnitude"), NULL},
	{"y", point_get_y, NULL, PyDoc_STR("The vertical magnitude"), NULL},
	{NULL}
};

static PyObj
point_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source, rob = NULL;

	if (!PyArg_ParseTuple(args, "O", &source))
		return(NULL);
	if (source->ob_type == subtype)
	{
		Py_INCREF(source);
		return(source);
	}

	rob = PyPg_point_FromPyObject(subtype, source);
	return(rob);
}

PyDoc_STRVAR(PyPg_point_Type_Doc, "Point interface type");

PyPgTypeObject PyPg_point_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"point",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&point_as_sequence,						/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_point_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	point_methods,								/* tp_methods */
	NULL,											/* tp_members */
	point_getset,								/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	point_new,									/* tp_new */
},
};
#endif /* POINT */

#ifdef PGTYPE_LSEG
PyObj
PyPg_lseg_FromPySequence(PyTypeObject *subtype, PyObj source)
{
	LSEG ls;
	PyObj p, x, y, sx, sy, rob;

	if (PySequence_Length(source) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"lseg requires a two-length sequence, given %d",
			PySequence_Length(source));
		return(NULL);
	}

	/* first point */
	p = PySequence_GetItem(source, 0);
	if (p == NULL) return(NULL);

	if (PySequence_Length(p) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"lseg point requires a two-length sequence, given %d",
			PySequence_Length(source));
		Py_DECREF(p);
		return(NULL);
	}

	sx = PySequence_GetItem(p, 0);
	if (sx == NULL)
	{
		Py_DECREF(p);
		return(NULL);
	}

	sy = PySequence_GetItem(p, 1);
	Py_DECREF(p);
	if (sy == NULL) return(NULL);

	x = PyNumber_Float(sx);
	Py_DECREF(sx);
	if (x == NULL) return(NULL);

	y = PyNumber_Float(sy);
	Py_DECREF(sy);
	if (y == NULL) return(NULL);

	ls.p[0].x = PyFloat_AsDouble(x);
	Py_DECREF(x);
	ls.p[0].y = PyFloat_AsDouble(y);
	Py_DECREF(y);

	/* second point */
	p = PySequence_GetItem(source, 1);
	if (p == NULL) return(NULL);

	if (PySequence_Length(p) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"lseg point requires a two-length sequence, given %d",
			PySequence_Length(source));
		Py_DECREF(p);
		return(NULL);
	}

	sx = PySequence_GetItem(p, 0);
	if (sx == NULL)
	{
		Py_DECREF(p);
		return(NULL);
	}

	sy = PySequence_GetItem(p, 1);
	Py_DECREF(p);
	if (sy == NULL) return(NULL);

	x = PyNumber_Float(sx);
	Py_DECREF(sx);
	if (x == NULL) return(NULL);

	y = PyNumber_Float(sy);
	Py_DECREF(sy);
	if (y == NULL) return(NULL);

	ls.p[1].x = PyFloat_AsDouble(x);
	Py_DECREF(x);
	ls.p[1].y = PyFloat_AsDouble(y);
	Py_DECREF(y);

	ls.m = 0;

	rob = PyPgObject_New(subtype, LsegPGetDatum(&ls));
	return(rob);
}

PyObj
PyPg_lseg_FromPyObject(PyTypeObject *subtype, PyObj source)
{
	PyObj rob;

	if (subtype == source->ob_type)
	{
		Py_INCREF(source);
		return(source);
	}

	if (PySequence_Check(source)
		&& !PyString_Check(source)
		&& !PyPg_text_CheckExact(source)
		&& !PyPg_varchar_CheckExact(source)
		&& !PyPg_cstring_CheckExact(source))
	{
		rob = PyPg_lseg_FromPySequence(subtype, source);
	}
	else
	{
		rob = PyPgObject_typinput((PyObj) subtype, source);
	}

	return(rob);
}

static int
lseg_sequence_length(PyObj self)
{
	return(2);
}

static PyObj
lseg_item(PyObj self, int item)
{
	LSEG *ls = DatumGetLsegP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(ls != NULL);

	if (item > 1)
	{
		PyErr_Format(PyExc_IndexError,
			"index %d is out of range(2)", item);
		return(NULL);
	}

	rob = PyPgObject_New(&PyPg_point_Type,
		PointPGetDatum(item == 0 ? &(ls->p[0]) : &(ls->p[1])));
	return(rob);
}

static PySequenceMethods lseg_as_sequence = {
	lseg_sequence_length,	/* sq_length */
	NULL,							/* sq_concat */
	NULL,							/* sq_repeat */
	lseg_item,					/* sq_item */
	NULL,							/* sq_slice */
	NULL,							/* sq_ass_item */
	NULL,							/* sq_ass_slice */
	NULL,							/* sq_contains */
	NULL,							/* sq_inplace_concat */
	NULL,							/* sq_inplace_repeat */
};

#define PyPg_bool_returning_lseg_Method(fname, pg_func) \
static PyObj fname(PyObj self, PyObj ob) \
{ \
	Datum rd = 0; \
	PyObj rob; \
\
	ob = PyPg_lseg_FromPyObject(self->ob_type, ob); \
	if (ob == NULL) return(NULL); \
\
	PgError_TRAP( \
		rd = DirectFunctionCall2(pg_func, \
			PyPgObject_FetchDatum(self), \
			PyPgObject_FetchDatum(ob) \
		) \
	); \
	Py_DECREF(ob); \
	if (PyErr_Occurred()) return(NULL); \
\
	rob = rd == 0 ? Py_False : Py_True; \
	Py_INCREF(rob); \
\
	return(rob); \
}

#define PyPg_point_returning_lseg_Method(fname, pg_func) \
static PyObj fname(PyObj self, PyObj ob) \
{ \
	Datum rd = 0; \
	PyObj rob; \
\
	ob = PyPg_lseg_FromPyObject(self->ob_type, ob); \
	if (ob == NULL) return(NULL); \
\
	PgError_TRAP( \
		rd = DirectFunctionCall2(pg_func, \
			PyPgObject_FetchDatum(self), \
			PyPgObject_FetchDatum(ob) \
		) \
	); \
	Py_DECREF(ob); \
	if (PyErr_Occurred()) return(NULL); \
\
	rob = PyPgObject_New(&PyPg_point_Type, rd); \
	pfree(DatumGetPointer(rd)); \
\
	return(rob); \
}

PyPg_bool_returning_lseg_Method(py_lseg_intersects, lseg_intersect);
PyPg_bool_returning_lseg_Method(py_lseg_parallel, lseg_parallel);
PyPg_bool_returning_lseg_Method(py_lseg_perpendicular, lseg_perp);

PyPg_point_returning_lseg_Method(py_lseg_intersection, lseg_interpt);

static PyMethodDef lseg_methods[] = {
	{"intersects", (PyCFunction) py_lseg_intersects, METH_O,
	PyDoc_STR("Whether the lines of intersect")},
	{"parallel", (PyCFunction) py_lseg_parallel, METH_O,
	PyDoc_STR("Whether the lines are parallel")},
	{"perpendicular", (PyCFunction) py_lseg_perpendicular, METH_O,
	PyDoc_STR("Whether the line is perpendicular to this line")},

	{"intersection", (PyCFunction) py_lseg_intersection, METH_O,
	PyDoc_STR("The point of intersection")},
	{NULL}
};

static PyObj
lseg_get_horizontal(PyObj self, void *closure)
{
	Datum rd = 0, ls = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(ls != 0);

	PgError_TRAP(rd = DirectFunctionCall1(lseg_horizontal, ls));
	if (PyErr_Occurred()) return(NULL);

	rob = rd == 0 ? Py_False : Py_True;
	return(rob);
}

static PyObj
lseg_get_vertical(PyObj self, void *closure)
{
	Datum rd = 0, ls = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(ls != 0);

	PgError_TRAP(rd = DirectFunctionCall1(lseg_vertical, ls));
	if (PyErr_Occurred()) return(NULL);

	rob = rd == 0 ? Py_False : Py_True;
	return(rob);
}


static PyObj
lseg_get_length(PyObj self, void *closure)
{
	Datum rd = 0, ls = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(ls != 0);

	PgError_TRAP(rd = DirectFunctionCall1(lseg_length, ls));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));
	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyGetSetDef lseg_getset[] = {
	{"length", lseg_get_length, NULL,
	PyDoc_STR("The length of the line segment"), NULL},
	{"vertical", lseg_get_vertical, NULL,
	PyDoc_STR("Whether the line is vertical"), NULL},
	{"horizontal", lseg_get_horizontal, NULL,
	PyDoc_STR("Whether the line is horizontal"), NULL},
	{NULL}
};

static PyObj
lseg_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source, rob = NULL;

	if (!PyArg_ParseTuple(args, "O", &source))
		return(NULL);
	if (source->ob_type == subtype)
	{
		Py_INCREF(source);
		return(source);
	}

	rob = PyPg_lseg_FromPyObject(subtype, source);
	return(rob);
}

PyDoc_STRVAR(PyPg_lseg_Type_Doc, "Line segment interface type");

PyPgTypeObject PyPg_lseg_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"lseg",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&lseg_as_sequence,						/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,			  		/* tp_flags */
	PyPg_lseg_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	lseg_methods,								/* tp_methods */
	NULL,											/* tp_members */
	lseg_getset,								/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	lseg_new,									/* tp_new */
},
};
#endif /* LSEG */

#ifdef PGTYPE_PATH
static int
path_sequence_length(PyObj self)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(path_npoints, d));
	if (PyErr_Occurred()) return(-1);

	return(DatumGetInt32(rd));
}

static PyObj
path_item(PyObj self, int item)
{
	PATH *path = DatumGetPathP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(path != NULL);

	if (item >= path->npts || item < 0)
	{
		PyErr_Format(PyExc_IndexError,
			"index %d is out of range(%d)", item, path->npts);
		return(NULL);
	}

	rob = PyPgObject_New(&PyPg_point_Type, PointPGetDatum(&(path->p[item])));
	return(rob);
}

static PySequenceMethods path_as_sequence = {
	path_sequence_length,	/* sq_length */
	NULL,							/* sq_concat */
	NULL,							/* sq_repeat */
	path_item,					/* sq_item */
	NULL,							/* sq_slice */
	NULL,							/* sq_ass_item */
	NULL,							/* sq_ass_slice */
	NULL,							/* sq_contains */
	NULL,							/* sq_inplace_concat */
	NULL,							/* sq_inplace_repeat */
};

static PyObj
path_get_points(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(path_npoints, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyInt_FromLong(DatumGetInt32(rd));

	return(rob);
}

static PyObj
path_get_center(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(path_center, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyPgObject_New(&PyPg_point_Type, rd);
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
path_get_area(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(path_area, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyGetSetDef path_getset[] = {
	{"points", path_get_points, NULL,
	PyDoc_STR("The number of points in the path"), NULL},
	{"center", path_get_center, NULL,
	PyDoc_STR("The center of the path"), NULL},
	{"area", path_get_area, NULL,
	PyDoc_STR("The area of the path"), NULL},
	{NULL}
};

PyDoc_STRVAR(PyPg_path_Type_Doc, "Path interface type");

PyPgTypeObject PyPg_path_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"path",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&path_as_sequence,						/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_path_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	path_getset,								/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* PATH */

#ifdef PGTYPE_BOX
PyObj
PyPg_box_FromPySequence(PyTypeObject *subtype, PyObj source)
{
	BOX b;
	Point high, low;
	PyObj p, x, y, sx, sy, rob;

	if (PySequence_Length(source) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"box requires a two-length sequence, given %d",
			PySequence_Length(source));
		return(NULL);
	}

	/* first point */
	p = PySequence_GetItem(source, 0);
	if (p == NULL) return(NULL);

	if (PySequence_Length(p) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"box point requires a two-length sequence, given %d",
			PySequence_Length(source));
		Py_DECREF(p);
		return(NULL);
	}

	sx = PySequence_GetItem(p, 0);
	if (sx == NULL)
	{
		Py_DECREF(p);
		return(NULL);
	}

	sy = PySequence_GetItem(p, 1);
	Py_DECREF(p);
	if (sy == NULL) return(NULL);

	x = PyNumber_Float(sx);
	Py_DECREF(sx);
	if (x == NULL) return(NULL);

	y = PyNumber_Float(sy);
	Py_DECREF(sy);
	if (y == NULL) return(NULL);

	high.x = PyFloat_AsDouble(x);
	Py_DECREF(x);
	high.y = PyFloat_AsDouble(y);
	Py_DECREF(y);

	/* second point */
	p = PySequence_GetItem(source, 1);
	if (p == NULL) return(NULL);

	if (PySequence_Length(p) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"box point requires a two-length sequence, given %d",
			PySequence_Length(source));
		Py_DECREF(p);
		return(NULL);
	}

	sx = PySequence_GetItem(p, 0);
	if (sx == NULL)
	{
		Py_DECREF(p);
		return(NULL);
	}

	sy = PySequence_GetItem(p, 1);
	Py_DECREF(p);
	if (sy == NULL) return(NULL);

	x = PyNumber_Float(sx);
	Py_DECREF(sx);
	if (x == NULL) return(NULL);

	y = PyNumber_Float(sy);
	Py_DECREF(sy);
	if (y == NULL) return(NULL);

	low.x = PyFloat_AsDouble(x);
	Py_DECREF(x);
	low.y = PyFloat_AsDouble(y);
	Py_DECREF(y);

	if (low.x > high.x)
	{
		b.high.x = low.x;
		b.low.x = high.x;
	}
	else
	{
		b.high.x = high.x;
		b.low.x = low.x;
	}

	if (low.y > high.y)
	{
		b.high.y = low.y;
		b.low.y = high.y;
	}
	else
	{
		b.high.y = high.y;
		b.low.y = low.y;
	}

	rob = PyPgObject_New(subtype, BoxPGetDatum(&b));
	return(rob);
}

PyObj
PyPg_box_FromPyObject(PyTypeObject *subtype, PyObj source)
{
	PyObj rob;

	if (subtype == source->ob_type)
	{
		Py_INCREF(source);
		return(source);
	}

	if (PySequence_Check(source)
		&& !PyString_Check(source)
		&& !PyPg_text_CheckExact(source)
		&& !PyPg_varchar_CheckExact(source)
		&& !PyPg_cstring_CheckExact(source))
	{
		rob = PyPg_box_FromPySequence(subtype, source);
	}
	else
	{
		rob = PyPgObject_typinput((PyObj) subtype, source);
	}

	return(rob);
}

static int
box_sequence_length(PyObj self)
{
	return(2);
}

static PyObj
box_item(PyObj self, int item)
{
	BOX *b = DatumGetBoxP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(b != NULL);

	if (item > 1 || item < 0)
	{
		PyErr_Format(PyExc_IndexError,
			"index %d is out of range(2)", item);
		return(NULL);
	}

	rob = PyPgObject_New(&PyPg_point_Type,
		PointPGetDatum(item == 0 ? &(b->high) : &(b->low)));
	return(rob);
}

static PySequenceMethods box_as_sequence = {
	box_sequence_length,		/* sq_length */
	NULL,							/* sq_concat */
	NULL,							/* sq_repeat */
	box_item,					/* sq_item */
	NULL,							/* sq_slice */
	NULL,							/* sq_ass_item */
	NULL,							/* sq_ass_slice */
	NULL,							/* sq_contains */
	NULL,							/* sq_inplace_concat */
	NULL,							/* sq_inplace_repeat */
};

static PyObj
box_get_high(PyObj self, void *closure)
{
	BOX *b = DatumGetBoxP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(b != NULL);

	rob = PyPgObject_New(&PyPg_point_Type, PointPGetDatum(&(b->high)));
	return(rob);
}

static PyObj
box_get_low(PyObj self, void *closure)
{
	BOX *b = DatumGetBoxP(PyPgObject_FetchDatum(self));
	PyObj rob;

	Assert(b != NULL);

	rob = PyPgObject_New(&PyPg_point_Type, PointPGetDatum(&(b->low)));
	return(rob);
}

static PyObj
box_get_height(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(box_height, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
box_get_width(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(box_width, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyGetSetDef box_getset[] = {
	{"high", box_get_high, NULL,
	PyDoc_STR("The high point of the box"), NULL},
	{"low", box_get_low, NULL,
	PyDoc_STR("The low point of the box"), NULL},
	{"height", box_get_height, NULL,
	PyDoc_STR("The height of the box"), NULL},
	{"width", box_get_width, NULL,
	PyDoc_STR("The width of the box"), NULL},
	{NULL}
};

static PyObj
box_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source, rob = NULL;

	if (!PyArg_ParseTuple(args, "O", &source))
		return(NULL);
	if (source->ob_type == subtype)
	{
		Py_INCREF(source);
		return(source);
	}

	rob = PyPg_box_FromPyObject(subtype, source);
	return(rob);
}

PyDoc_STRVAR(PyPg_box_Type_Doc, "Box interface type");

PyPgTypeObject PyPg_box_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"box",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&box_as_sequence,							/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_box_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	box_getset,									/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	box_new,										/* tp_new */
},
};
#endif /* BOX */

#ifdef PGTYPE_LINE
PyDoc_STRVAR(PyPg_line_Type_Doc, "Line interface type");

PyPgTypeObject PyPg_line_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"line",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_line_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* LINE */

#ifdef PGTYPE_CIRCLE
static PyObj
circle_get_center(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(circle_center, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyPgObject_New(&PyPg_point_Type, rd);

	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
circle_get_radius(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(circle_radius, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
circle_get_diameter(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(circle_diameter, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyObj
circle_get_circumference(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;
	double circ;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(circle_diameter, d));
	if (PyErr_Occurred()) return(NULL);
	circ = DatumGetFloat8(rd) * M_PI;

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	rob = PyFloat_FromDouble(circ);

	return(rob);
}

static PyObj
circle_get_area(PyObj self, void *closure)
{
	Datum rd = 0, d = PyPgObject_FetchDatum(self);
	PyObj rob;

	Assert(d != 0);

	PgError_TRAP(rd = DirectFunctionCall1(circle_area, d));
	if (PyErr_Occurred()) return(NULL);
	rob = PyFloat_FromDouble(DatumGetFloat8(rd));

	/* XXX: if !typbyval */
	pfree(DatumGetPointer(rd));

	return(rob);
}

static PyGetSetDef circle_getset[] = {
	{"center", circle_get_center, NULL,
	PyDoc_STR("The circle's center point"), NULL},
	{"radius", circle_get_radius, NULL,
	PyDoc_STR("The circle's radius"), NULL},
	{"diameter", circle_get_diameter, NULL,
	PyDoc_STR("The circle's diameter"), NULL},
	{"circumference", circle_get_circumference, NULL,
	PyDoc_STR("The circle's circumference"), NULL},
	{"area", circle_get_area, NULL,
	PyDoc_STR("The circle's area"), NULL},
	{NULL}
};

PyDoc_STRVAR(PyPg_circle_Type_Doc, "Circle interface type");

PyPgTypeObject PyPg_circle_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"circle",									/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT,		   		/* tp_flags */
	PyPg_circle_Type_Doc,					/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	circle_getset,								/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	PyPgArbitrary_new,						/* tp_new */
},
};
#endif /* CIRCLE */

/*
 * vim: ts=3:sw=3:noet:
 */
