
/* psqlkoi8.c -- perform character translation from koi8 to 1251 encoding.
 *
 * This code is in the public domain.
 *
 * Author: Aleksey Demakov <avd@gcom.ru>
 */

#include <windows.h>
#include <sql.h>
#include <odbcinst.h>
#include <string.h>

#define DATATRUNCATED "[PostgreSQL KOI8 Translator] Data was truncated"

static unsigned char kw[] = {
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  254,224,225,246,228,229,244,227,245,232,233,234,235,236,237,238,
  239,255,240,241,242,243,230,226,252,251,231,248,253,249,247,250,
  222,192,193,214,196,197,212,195,213,200,201,202,203,204,205,206,
  207,223,208,209,210,211,198,194,220,219,199,216,221,217,215,218};

static unsigned char wk[] = {
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  225,226,247,231,228,229,246,250,233,234,235,236,237,238,239,240,
  242,243,244,245,230,232,227,254,251,253,255,249,248,252,224,241,
  193,194,215,199,196,197,214,218,201,202,203,204,205,206,207,208,
  210,211,212,213,198,200,195,222,219,221,223,217,216,220,192,209};

BOOL WINAPI
ConfigTranslator (HWND hwndParent,
		  DWORD FAR *pvOption)
{
  *pvOption = 0;
  return TRUE;
}

void
Error (const char *msg,
       UCHAR FAR *szErrorMsg,
       SWORD cbErrorMsg,
       SWORD FAR *pcbErrorMsg)

{
  if (szErrorMsg != NULL)
    {
      int n = strlen (msg);
      if (n < cbErrorMsg)
	strcpy (szErrorMsg, msg);
      else
	{
	  memcpy (szErrorMsg, msg, cbErrorMsg - 1);
	  szErrorMsg[cbErrorMsg - 1] = 0;
	}
      if (pcbErrorMsg != NULL)
	*pcbErrorMsg = n;
    }
}

BOOL
Translate (unsigned char *xlat,
	   SWORD fSqlType,
	   PTR rgbValueIn,
	   SDWORD cbValueIn,
	   PTR rgbValueOut,
	   SDWORD cbValueOutMax,
	   SDWORD FAR *pcbValueOut,
	   UCHAR FAR *szErrorMsg,
	   SWORD cbErrorMsg,
	   SWORD FAR *pcbErrorMsg)
{
  int i, n;
  if (fSqlType == SQL_CHAR
      || fSqlType == SQL_VARCHAR
      || fSqlType == SQL_LONGVARCHAR)
    {
      if (cbValueIn == SQL_NTS)
	cbValueIn = strlen (rgbValueIn);
      n = cbValueIn < cbValueOutMax ? cbValueIn : cbValueOutMax;
      for (i = 0; i < n; i++)
	{
	  int c = ((UCHAR *) rgbValueIn)[i];
	  ((UCHAR *) rgbValueOut)[i] = c & 0x80 ? xlat[c & 0x7f] : c;
	}
#if 0
      if (i < cbValueOutMax)
	((UCHAR *) rgbValueOut)[i] = 0;
#endif
      if (n <= cbValueIn)
	{
	  Error (DATATRUNCATED, szErrorMsg, cbErrorMsg, pcbErrorMsg);
	  return FALSE;
	}
    }
  else if (rgbValueIn != rgbValueOut)
    {
      if (fSqlType == SQL_BINARY
	  || fSqlType == SQL_VARBINARY
	  || fSqlType == SQL_LONGVARBINARY)
	{
	  n = cbValueIn < cbValueOutMax ? cbValueIn : cbValueOutMax;
	  memcpy (rgbValueOut, rgbValueIn, n);
	  if (n <= cbValueIn)
	    {
	      Error (DATATRUNCATED, szErrorMsg, cbErrorMsg, pcbErrorMsg);
	      return FALSE;
	    }
	}
      else
	{
	  switch (fSqlType)
	    {
	    case SQL_DECIMAL:
	    case SQL_NUMERIC:
	      /* FIXME */
	      n = cbValueIn;
	      break;
	    case SQL_BIT:
	      n = sizeof (UCHAR);
	      break;
	    case SQL_TINYINT:
	      n = sizeof (SCHAR);
	      break;
	    case SQL_SMALLINT:
	      n = sizeof (SWORD);
	      break;
	    case SQL_INTEGER:
	      n = sizeof (SDWORD);
	      break;
	    case SQL_BIGINT:
	      /* FIXME */
	      n = cbValueIn;
	      break;
	    case SQL_REAL:
	      n = sizeof (SFLOAT);
	      break;
	    case SQL_FLOAT:
	    case SQL_DOUBLE:
	      n = sizeof (SDOUBLE);
	      break;
	    case SQL_DATE:
	      n = sizeof (DATE_STRUCT);
	      break;
	    case SQL_TIME:
	      n = sizeof (TIME_STRUCT);
	      break;
	    case SQL_TIMESTAMP:
	      n = sizeof (TIMESTAMP_STRUCT);
	      break;
	    }
	  memcpy (rgbValueOut, rgbValueIn, n);
	}
    }

  if (pcbValueOut != NULL)
    *pcbValueOut = cbValueIn;

  return TRUE;
}

BOOL WINAPI
SQLDataSourceToDriver (UDWORD fOption,
		       SWORD fSqlType,
		       PTR rgbValueIn,
		       SDWORD cbValueIn,
		       PTR rgbValueOut,
		       SDWORD cbValueOutMax,
		       SDWORD FAR *pcbValueOut,
		       UCHAR FAR *szErrorMsg,
		       SWORD cbErrorMsg,
		       SWORD FAR *pcbErrorMsg)
{
  return Translate (kw,
		    fSqlType,
		    rgbValueIn, cbValueIn,
		    rgbValueOut, cbValueOutMax, pcbValueOut,
		    szErrorMsg, cbErrorMsg, pcbErrorMsg);
}

BOOL WINAPI
SQLDriverToDataSource(UDWORD fOption,
		      SWORD fSqlType,
		      PTR rgbValueIn,
		      SDWORD cbValueIn,
		      PTR  rgbValueOut,
		      SDWORD cbValueOutMax,
		      SDWORD FAR *pcbValueOut,
		      UCHAR FAR *szErrorMsg,
		      SWORD cbErrorMsg,
		      SWORD FAR *pcbErrorMsg)
{
  return Translate (wk,
		    fSqlType,
		    rgbValueIn, cbValueIn,
		    rgbValueOut, cbValueOutMax, pcbValueOut,
		    szErrorMsg, cbErrorMsg, pcbErrorMsg);
}
