/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbtablemanager.h"
#include "dbtablecolumn.h"
#include "./indicies/dbindexset.h"
#include "./indicies/dbindexmanager.h"
#include "../../utils/converter.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"
#include <stdlib.h>

	/**
 	 * Constructor
	 */
  DBTableManager::DBTableManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBTableManager::DBTableManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBTableManager::~DBTableManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to create a new table.
	 * @param			roTableSchema		an object representing the new table to be created
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::createTableSchema(const DBTableSchema & roTableSchema)
				throw (SQLException, DBConnectionException)
	{
		// need to build sql create statement
		string strSQL = "CREATE TABLE " + roTableSchema.getTableName() + "(";
		string strPrimaryKey = "";
		// go through columns and add properties
		for (int nColumnIdx = 0; nColumnIdx < roTableSchema.getNumberOfColumns(); nColumnIdx++)
		{
			DBTableColumn oTableColumn = roTableSchema.getColumn(nColumnIdx);
			// add column name
			if (nColumnIdx == 0)
			{
				strSQL += oTableColumn.getFieldName();
			} // end if first column
			else
			{
				strSQL += ", " + oTableColumn.getFieldName();
			} // end else not first column
			
			// add sql type
			strSQL += " " + oTableColumn.getFieldType();
			if (oTableColumn.getFieldSize() != 0)
			{
				strSQL += "(" + Converter::intToString(oTableColumn.getFieldSize()) + ")";
			} // end if field size is set
			else if (oTableColumn.isNumeric())
			{
				strSQL += "(" + Converter::intToString(oTableColumn.getNumericPrecision())
											+ ", " + Converter::intToString(oTableColumn.getNumericDecimal()) + ")";
			} // end else field is numeric
			
			// add not null clause
			if (oTableColumn.isFieldNullable() == false)
			{
				strSQL += " NOT NULL";
			} // end if field is nullable
			
			// add default clause
			if (oTableColumn.getFieldDefault() != "")
			{
				strSQL += " DEFAULT '" + StringUtils::databasestr(oTableColumn.getFieldDefault()) + "'";
			} // end if field has a default
			
			// add to primary key
			if (oTableColumn.isPrimaryKey())
			{
				if (strPrimaryKey == "")
				{
					strPrimaryKey = oTableColumn.getFieldName();
				} // end if no primary key set
				else
				{
					strPrimaryKey += ", " + oTableColumn.getFieldName();
				} // end else primary key has been set
			} // end if primary key
		} // end for more columns to add
		
		// add primary key clause
		if (strPrimaryKey != "")
		{
			strSQL += ", PRIMARY KEY (" + strPrimaryKey + ")";
		} // end if primary key exists
		
		// add end bracket
		strSQL += ")";
		
		// execute create table
		m_poDBConn->execute(strSQL);	
		
		// loop through comments and add comments
		for (int nColumnIdx = 0; nColumnIdx < roTableSchema.getNumberOfColumns(); nColumnIdx++)
		{
			string strSQLComment;
			DBTableColumn oTableColumn = roTableSchema.getColumn(nColumnIdx);
			// if comment exists
			if (oTableColumn.getFieldComment() != "")
			{
				strSQLComment = "COMMENT ON COLUMN " + roTableSchema.getTableName() + "." +
														oTableColumn.getFieldName() + " IS '" +
														StringUtils::databasestr(oTableColumn.getFieldComment()) + "'";
				// create comment
				m_poDBConn->execute(strSQLComment);	
			} // end if comments exist
		} // end for more columns with comments
														
	} // end createTableSchema

	/**
	 * Used to drop a table/view.
	 * @param			rstrTableName		the name of the table/view to drop	
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::dropTableSchema(const string & rstrTableName)
				throw (SQLException, DBConnectionException)
	{
		string strSQLDrop = "DROP TABLE " + rstrTableName;
		// drop table
		m_poDBConn->execute(strSQLDrop);												
		
	} // end dropTableSchema

	/**
	 * Used to modify the table comment.
	 * @param			rstrTableName		the name of the table
	 * @param			rstrComment			the comment to set for the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::modifyTableComment(const string & rstrTableName, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON TABLE " + rstrTableName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON TABLE " + rstrTableName +
 												" IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
	
	} // end modifyTableComment
		
	/**
	 * Used to add a table column.
	 * Currently constraints are not supported. Column defaults are supported.
	 * @param			rstrTableName		the table the column belong to
	 * @param			roTableColNew		the new column to add to the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::addTableColumn(const string & rstrTableName, const DBTableColumn & roTableColNew)
				throw (SQLException, DBConnectionException)
	{
		// add column to table
		string strSQLAdd = "ALTER TABLE " + rstrTableName + " ADD COLUMN " +
													roTableColNew.getFieldName() + " " + roTableColNew.getFieldType();
 		// add sql type
 		if (roTableColNew.getFieldSize() != 0)
 		{
 			strSQLAdd += "(" + Converter::intToString(roTableColNew.getFieldSize()) + ")";
 		} // end if field size is set
 		else if (roTableColNew.isNumeric())
 		{
 			strSQLAdd += "(" + Converter::intToString(roTableColNew.getNumericPrecision())
 											 + ", " + Converter::intToString(roTableColNew.getNumericDecimal()) + ")";
 		} // end else field is numeric
		// add column
		m_poDBConn->execute(strSQLAdd);												
		
		// check to see if column default exists
		if (roTableColNew.getFieldDefault() != "")
		{
			// alter column default							
			string strSQLAlter = "ALTER TABLE " + rstrTableName +
														" ALTER COLUMN " + roTableColNew.getFieldName() +
														" SET DEFAULT '" + StringUtils::databasestr(roTableColNew.getFieldDefault()) + "'";
			// set default
			m_poDBConn->execute(strSQLAlter);														
		} // end if default exists	
		
 		// update comment
 		modifyColumnComment(rstrTableName, roTableColNew.getFieldName(), roTableColNew.getFieldComment());
							
	} // end addTableColumn			

	/**
	 * Used to alter the table name.
	 * @param			rstrNewTableName		the new table name
	 * @param			rstrOldTableName		the old table name
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::alterTableName(const string & rstrNewTableName, const string & rstrOldTableName)
				throw (SQLException, DBConnectionException)
	{
		string strSQLAlter;
		// check to see if the table name has really changed
		if (rstrNewTableName != rstrOldTableName)
		{
			strSQLAlter = "ALTER TABLE " + rstrOldTableName +
											" RENAME TO " + rstrNewTableName;
			// rename column
			m_poDBConn->execute(strSQLAlter);												
		} // end if table name has changed	
	} // end alterTableName
						
	/**
	 * Used to alter a table column.
	 * Currently the features supported by alter:
	 * - Change column name
	 * - Change the default of the column
	 * - Change the column comment	
	 * @param			rstrTableName		the table the columns belong to
	 * @param			roTableColNew		the new version of the column
	 * @param			roTableColOld		the old version of the column
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::alterTableColumn(const string & rstrTableName, const DBTableColumn & roTableColNew,
				const DBTableColumn & roTableColOld)
				throw (SQLException, DBConnectionException)
	{
		string strSQLAlter;
		// check to see if the column name has changed
		if (roTableColNew.getFieldName() != roTableColOld.getFieldName())
		{
			strSQLAlter = "ALTER TABLE " + rstrTableName +
											" RENAME COLUMN " + roTableColOld.getFieldName() +
											" TO " + roTableColNew.getFieldName();
			// rename column
			m_poDBConn->execute(strSQLAlter);												
		} // end if column name has changed

		// check to see if the default has changed
		if (roTableColNew.getFieldDefault() != roTableColOld.getFieldDefault())
		{
			if (roTableColNew.getFieldDefault() == "")
			{
				strSQLAlter = "ALTER TABLE " + rstrTableName +
												" ALTER COLUMN " + roTableColNew.getFieldName() +
												" DROP DEFAULT";				
			} // end if need to drop default
			else
			{
				strSQLAlter = "ALTER TABLE " + rstrTableName +
												" ALTER COLUMN " + roTableColNew.getFieldName() +
												" SET DEFAULT '" + StringUtils::databasestr(roTableColNew.getFieldDefault()) + "'";
			} // end else need to set new default											
			// rename column
			m_poDBConn->execute(strSQLAlter);												
		} // end if column name has changed
		
		// check to see if the comment has changed
		if (roTableColNew.getFieldComment() != roTableColOld.getFieldComment())
		{
			// update comment
			modifyColumnComment(rstrTableName, roTableColNew.getFieldName(), roTableColNew.getFieldComment());
		} // end if comment has changed
				
	} // end alterTableColumn

	/**
	 * Used to modify the column comment.
	 * @param			rstrTableName		the name of the table
	 * @param			rstrColumnName	the name of the column	
	 * @param			rstrComment			the comment to set for the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::modifyColumnComment(const string & rstrTableName, const string & rstrColumnName,
				const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON COLUMN " + rstrTableName + "." +
 												rstrColumnName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON COLUMN " + rstrTableName + "." +
 												rstrColumnName + " IS '" +
 												StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
 		// change comment on column
 		m_poDBConn->execute(strSQLComment);														
	
	} // end modifyColumnComment
											
	/**
	 * Used to retrieve the a table schema.
	 * @param			rstrTableName
	 * @param			roTableSchema
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::retrieveTableSchema(const string & rstrTableName, DBTableSchema & roTableSchema)
				throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBTableManager::retrieveTableSchema";
		Debugger::entered(strMethodName);
		
		// sql statement used to retrieve all the table schema details
		string strSQL = "SELECT a.attnum, a.attname as fieldname, t.typname as type, a.attlen as length, "
													 "a.atttypmod as lengthvar, a.attnotnull as notnull, "
													 "d.adsrc as default "
										"FROM pg_class c, pg_attribute a, pg_type t, pg_attrdef d "
										"WHERE c.relname = '" + rstrTableName + "' "
										"AND a.attnum > 0 "
										"AND a.attrelid = c.oid "
										"AND a.atttypid = t.oid "
										"AND d.adrelid = c.oid "
										"AND d.adnum = a.attnum "
										"UNION ALL "
										"SELECT a.attnum, a.attname as fieldname, t.typname as type, a.attlen as length, "
													 "a.atttypmod as lengthvar, a.attnotnull as notnull, "
													 "NULL as default "
										"FROM pg_class c, pg_attribute a, pg_type t "
										"WHERE c.relname = '" + rstrTableName + "' "
										"AND a.attnum > 0 "
										"AND a.attrelid = c.oid "
										"AND a.atttypid = t.oid "
										"AND a.attnum NOT IN "
										"(select adnum from pg_attrdef d where d.adrelid = c.oid) "
										"ORDER BY attnum";

		// sql used to retrieve comment details										
		string strSQL1 = "SELECT a.attnum, a.attname as fieldname, d.description as comment "
										 "FROM pg_class c, pg_attribute a, pg_description d "
										 "WHERE c.relname = '" + rstrTableName + "' "
										 "AND a.attnum > 0 "
										 "AND a.attrelid = c.oid "
										 "AND d.objoid = a.oid "
										 "UNION ALL "
										 "SELECT a.attnum, a.attname as fieldname, NULL as comment "
										 "FROM pg_class c, pg_attribute a "
										 "WHERE c.relname = '" + rstrTableName + "' "
										 "AND a.attnum > 0 "
										 "AND a.attrelid = c.oid "
										 "AND a.oid NOT IN "
										 "(select objoid from pg_description d) "
										 "ORDER BY attnum";
										
		DBRecordSet rsTableSchema;										
		DBRecordSet rsFieldComments;
		string strFieldName, strTypeName, strLength, strFieldSize, strNotNull, strDefault, strComment;
		long lNumericSize = 0;
		int nLength = 0;
		int nFieldSize = 0;
		int nPrecision = 0;
		int nDecimal = 0;
		bool bNullable = false;
		bool bIsPrimaryKey = false;
		bool bPrimaryIndexExists = false;
		DBIndexSet oDBIndexSet;		
		DBIndexManager oDBIndexMgr;
		DBIndex oDBIndex;				
		vector<string> vstrColumns;
		DBTableColumn oDBTableCol;
				
		// execute query for schema details										
		m_poDBConn->executeQuery(strSQL, rsTableSchema);												
		// execute query for field comments
		m_poDBConn->executeQuery(strSQL1, rsFieldComments);		
		
		// also need to identify all primary keys
		oDBIndexMgr.setDBConnection(m_poDBConn);
		oDBIndexMgr.retrieveListOfIndicies(rstrTableName, oDBIndexSet, true);
		if (oDBIndexSet.next())
		{		
			oDBIndexSet.getDBIndex(oDBIndex);
			bPrimaryIndexExists = true;					
		} // end if primary index exists
		
		// make sure input table schema is cleared first before any new data is set
		roTableSchema.clear();
		
		// loop through recordset and assign values into table schema
		roTableSchema.setTableName(rstrTableName);
		while (rsTableSchema.next())
		{				
				// also move to comment field
				rsFieldComments.next();
				
				strFieldName = rsTableSchema.getFieldValue("fieldname");
			  strTypeName = rsTableSchema.getFieldValue("type");
			  strComment = rsFieldComments.getFieldValue("comment");
			  strNotNull = rsTableSchema.getFieldValue("notnull");
				strLength = rsTableSchema.getFieldValue("length");
			  strFieldSize = rsTableSchema.getFieldValue("lengthvar");
				// lets get rid of any slashes			  									
			  strDefault = StringUtils::stripSlashes(rsTableSchema.getFieldValue("default"));			
			  			
			  // if type is bpchar convert to char
			  if (strTypeName == "bpchar")
			  {
			  	strTypeName = "char";
			  } // end if bpchar			
			  else if (strTypeName == "int4")
			  {
			  	strTypeName = "integer";
			  } // end if integer
			
			  // find out if field is a primary key
			  bIsPrimaryKey = false;
			  if (bPrimaryIndexExists == true)
			  {
					for (int nIdx = 0; nIdx < oDBIndex.getNumberOfColumns(); nIdx++)
					{
						if (oDBIndex.getIndexedColumn(nIdx) == strFieldName)
						{
							bIsPrimaryKey = true;
							break;
						} // end if match
					} // end for more columns					
				} // end if primary index exists
				
			  // convert to boolean
			  if (strNotNull == DBConnection::DB_TRUE)
			  {
			  	bNullable = false;
			  } // end if not nullable
			  else
			  {
			  	bNullable = true;			  	
			  } // end else nullable

			  // check for numeric
			  if (strTypeName == DBTableColumn::NUMERIC_FIELD_NAME)
			  {
					lNumericSize = atol(strFieldSize.c_str());
					Debugger::logTrace(strMethodName, "Numeric field size = " + strFieldSize);	  	
					nPrecision = (lNumericSize >> 16) & 0xffff;
					Debugger::logTrace(strMethodName, "Precision part = " + Converter::intToString(nPrecision));	  						
					nDecimal = (lNumericSize - 4) & 0xffff;
					Debugger::logTrace(strMethodName, "Precision part = " + Converter::intToString(nDecimal));	  											
					
					// add numeric field
					oDBTableCol.setNumericFieldDetails(strFieldName, nPrecision, nDecimal, bIsPrimaryKey,
																							bNullable, strDefault, strComment);
			  } // end if numeric
			  else
			  {
				  // convert both to integers
				  nLength = atoi(strLength.c_str());
				  nFieldSize = atoi(strFieldSize.c_str());
				  if (nFieldSize == -1 && nLength == -1)
				  {
				  	nFieldSize = 0;
				  } // end if no size
				  else
				  {
				  	if (nLength == -1)
				  	{
				  		nFieldSize -= 4;	// subtract 4 bytes for all variable fields				  		
				  	} // end if size is not fixed
				  	else
				  	{
				  		nFieldSize = nLength;
				  	} // else fixed length type
				  } // end else has size				
				
				  // add non-numeric field
					oDBTableCol.setFieldDetails(strFieldName, strTypeName, nFieldSize, bIsPrimaryKey,
																							bNullable, strDefault, strComment);				
			  } // else not numeric
			
			  // add column to schema
			  roTableSchema.addColumn(oDBTableCol);
			} // end while more fields
		
		Debugger::exited(strMethodName);									
	} // end retrieveTableSchema
			
	/**
	 * Used to retrieve the list of tables for this connection.
	 * @param	a DBTableSet object that contains the table details
	 * @param	the string parameter that is passed to the like clause
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTableManager::retrieveListOfTables(DBTableSet & roDBTableSet, const string & rstrLikeClause = "")
				throw (SQLException, DBConnectionException)
	{
		// sql statement used to retrieve all the table details
		string strSQL = "SELECT t.tablename, t.tableowner, t.hasindexes, t.hasrules, "
													 "t.hastriggers, d.description "
										"FROM pg_tables t, pg_class c, pg_description d "
										"WHERE t.tablename = c.relname "
										"AND c.oid = d.objoid "
										"AND t.tablename not like 'pg\\_%' ";
										
		if (rstrLikeClause != "")										
		{
			strSQL += "AND t.tablename like '" + rstrLikeClause + "' ";
		} // end if like clause is not required
		
		strSQL +=				"UNION ALL "
										"SELECT t.tablename, t.tableowner, t.hasindexes, t.hasrules, "
													 "t.hastriggers, NULL "
										"FROM pg_tables t, pg_class c "
										"WHERE t.tablename = c.relname "
										"AND c.oid NOT IN (select objoid from pg_description) "
										"AND t.tablename not like 'pg\\_%' ";
		if (rstrLikeClause != "")										
		{
			strSQL += "AND t.tablename like '" + rstrLikeClause + "' ";
		} // end if like clause is not required
										
		strSQL +=				"ORDER BY tablename";
										
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBTableSet.m_oTableList);										
	} // end retrieveListOfTables
	
