/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbforeignkeyset.h"
#include <vector>
#include <string>
#include "../../../utils/stringutils.h"
#include "../../../utils/debugger.h"
#include "../../../utils/converter.h"

	/**
 	 * Constructor 	
   */		
  DBForeignKeySet::DBForeignKeySet()
  {
  	// do nothing
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBForeignKeySet::~DBForeignKeySet()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBForeignKeySet::isEmpty()
	{
		return m_oDBTriggerSet.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBForeignKeySet::getCurrentIndex()
	{
		return m_oDBTriggerSet.getCurrentIndex();
	} // end getCurrentTrigger
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBForeignKeySet::next()
	{
		return m_oDBTriggerSet.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBForeignKeySet::previous()
	{
		return m_oDBTriggerSet.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBForeignKeySet::reset()
	{
		m_oDBTriggerSet.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBForeignKeySet::first() throw (NoRecordException)
	{
		return m_oDBTriggerSet.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBForeignKeySet::last() throw (NoRecordException)
	{
		return m_oDBTriggerSet.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBForeignKeySet::getRecordCount()
	{
		return m_oDBTriggerSet.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current foreign key.
	 * @param	roDBForeignKey a DBForeignKey object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBForeignKeySet::getDBForeignKey(DBForeignKey & roDBForeignKey) throw (NoRecordException)
	{
		string strMethodName = "DBForeignKeySet::getDBForeignKey";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException("Cannot retrieve DBForeignKey object, no records exist.", "DBForeignKeySet", "getDBForeignKey");
		} // end if empty
		try
		{
			// get trigger
			DBTrigger oDBTrigger;
			m_oDBTriggerSet.getDBTrigger(oDBTrigger);
			
			roDBForeignKey.clear();		// lets make sure all info is cleared out
			// get foreign key info
			roDBForeignKey.setForeignKeyName(oDBTrigger.getTriggerArgument(0));		// this is the trigger name
			roDBForeignKey.setTableName(oDBTrigger.getTriggerArgument(1));
			roDBForeignKey.setForeignTableName(oDBTrigger.getTriggerArgument(2));
			// skip over index 3 - DON'T KNOW WHAT ITS FOR?
			
			// now lets loop through the rest and get columns that make up foreign key
			for (int nArgIdx = 4; nArgIdx < oDBTrigger.getNumberOfArguments(); nArgIdx = nArgIdx + 2)
			{
				roDBForeignKey.addReferencingColumn(oDBTrigger.getTriggerArgument(nArgIdx));
				roDBForeignKey.addForeignColumn(oDBTrigger.getTriggerArgument(nArgIdx+1));
			} // end for more args - increment by 2
			
		} // end try
		catch (VPException e)
		{
			Debugger::logException(strMethodName, e);
		} // end catch	
		
		Debugger::exited(strMethodName);
	} // end getDBTrigger

