/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbtriggerset.h"
#include <vector>
#include <string>
#include "../../../utils/stringutils.h"
#include "../../../utils/debugger.h"
#include "../../../utils/converter.h"
#include "../../dbconnection.h"

	/**
 	 * Constructor 	
   */		
  DBTriggerSet::DBTriggerSet()
  {
  	m_strClassName = "DBTriggerSet";
  	// do nothing
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBTriggerSet::~DBTriggerSet()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBTriggerSet::isEmpty()
	{
		return m_oTriggerList.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBTriggerSet::getCurrentIndex()
	{
		return m_oTriggerList.getCurrentIndex();
	} // end getCurrentTrigger
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBTriggerSet::next()
	{
		return m_oTriggerList.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBTriggerSet::previous()
	{
		return m_oTriggerList.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBTriggerSet::reset()
	{
		m_oTriggerList.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBTriggerSet::first() throw (NoRecordException)
	{
		return m_oTriggerList.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBTriggerSet::last() throw (NoRecordException)
	{
		return m_oTriggerList.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBTriggerSet::getRecordCount()
	{
		return m_oTriggerList.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current index.
	 * @param	roDBTrigger a DBTrigger object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBTriggerSet::getDBTrigger(DBTrigger & roDBTrigger) throw (NoRecordException)
	{
		string strMethodName = m_strClassName + "::getDBTrigger";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException("Cannot retrieve DBTrigger object, no records exist.", "DBTriggerSet", "getDBTrigger");
		} // end if empty
		try
		{
			roDBTrigger.setTableName(m_oTriggerList.getFieldValue("tablename"));
			roDBTrigger.setTriggerName(m_oTriggerList.getFieldValue("triggername"));			
			if (m_oTriggerList.getFieldValue("isconstraint") == DBConnection::DB_TRUE)
			{
				Debugger::logTrace(strMethodName, "Is constraint.");
				roDBTrigger.setConstraint(true);						
			} // end if constraint
			else
			{
				Debugger::logTrace(strMethodName, "Is NOT constraint.");			
				roDBTrigger.setConstraint(false);									
			} // end else not constraint
			string strTriggerType = m_oTriggerList.getFieldValue("triggertype");
			
			// convert to integer			
			roDBTrigger.setTriggerType(Converter::stringToInt(strTriggerType));			
			roDBTrigger.setProcedureName(m_oTriggerList.getFieldValue("procedurename"));
			roDBTrigger.setTriggerComment(m_oTriggerList.getFieldValue("comment"));			
			
			// lets get the arguments
			vector<string> vstrArgs;
			if (m_oTriggerList.getFieldValue("triggerargs").size() > 0)
			{
				Debugger::logTrace(strMethodName, "Trigger arguments retrieved: " + m_oTriggerList.getFieldValue("triggerargs"));			
				// lets break up arguments
				StringUtils::tokenize(m_oTriggerList.getFieldValue("triggerargs"), "\\000", vstrArgs);
				// make sure we ignore the last one, this is just empty
				for (int nArgsIdx = 0; nArgsIdx < vstrArgs.size() - 1; nArgsIdx++)
				{
					Debugger::logTrace(strMethodName, "Adding argument: " + vstrArgs[nArgsIdx]);
					roDBTrigger.addTriggerArgument(vstrArgs[nArgsIdx]);
				} // end for more arguments
			} // end if arguments exist
		} // end try
		catch (VPException e)
		{
		} // end catch	
		
		Debugger::exited(strMethodName);
	} // end getDBTrigger

