/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbsequencemanager.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"

	/**
 	 * Constructor
	 */
  DBSequenceManager::DBSequenceManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBSequenceManager::DBSequenceManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBSequenceManager::~DBSequenceManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to retrieve the list of sequences for this connection.
	 * @return	a DBSequenceSet object that contains the sequence details
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if not connected
	 */
	void DBSequenceManager::retrieveListOfSequences(DBSequenceSet & roDBSequenceSet)
				throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBViewManager::retrieveListOfSequences";
		
		Debugger::entered(strMethodName);
		
		// sql statement used to retrieve all the sequence details
		string strSQL = "SELECT c.relname as seqname, u.usename as seqowner, "
													 "d.description as comment "
										"FROM pg_class c, pg_user u, pg_description d "
										"WHERE c.relowner = u.usesysid "
										"AND c.oid = d.objoid "
										"AND c.relkind = 'S' "
										"AND u.usesuper = 'f' "
										"UNION ALL "
										"SELECT c.relname as seqname, u.usename as seqowner, "
													 "NULL as comment "
										"FROM pg_class c, pg_user u "
										"WHERE c.relowner = u.usesysid "
										"AND c.oid NOT IN (select objoid from pg_description) "
										"AND c.relkind = 'S' "										
										"AND u.usesuper = 'f' "										
										"ORDER BY seqname";
										
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBSequenceSet.m_oSequenceList);										
		
		Debugger::exited(strMethodName);
		
	} // end retrieveListOfSequences
	
	/**
	 * Used to modify the sequence comment.
	 * @param			rstrSequenceame		the name of the sequence
	 * @param			rstrComment				the comment to set for the sequence
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBSequenceManager::modifySequenceComment(const string & rstrSequenceName, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON SEQUENCE " + rstrSequenceName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON SEQUENCE " + rstrSequenceName +
 												" IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
		
	} // end modifySequenceComment			
