/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */
 
#include "dbtypemanager.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"
#include "../../utils/converter.h"

	/**
 	 * Constructor
	 */
  DBTypeManager::DBTypeManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBTypeManager::DBTypeManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBTypeManager::~DBTypeManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to retrieve the list of types based on the type oids passed in. The type
	 * oids are passed in via the DBType objects held in the vector.
	 * @param			rvoDBTypes (contains a vector of DBType objects)
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTypeManager::retrieveListOfTypes(vector<DBType> & rvoDBTypes)
		throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBTypeManager::retrieveListOfTypes";
		
		Debugger::entered(strMethodName);
		
		vector<long> voUncached;		
		// lets go through the list and see what's already cached and what's not
		for (int nIdx = 0; nIdx < rvoDBTypes.size(); nIdx++)
		{
			DBType oDBType = m_oDBTypeCache[rvoDBTypes[nIdx].getTypeOid()];
			// lets check to see if the type has been cached yet
			if (oDBType.getTypeOid() != rvoDBTypes[nIdx].getTypeOid())
			{
				Debugger::logTrace(strMethodName, "The Type Oid " +
									Converter::longToString(rvoDBTypes[nIdx].getTypeOid()) + " not yet cached.");
				// lets get the type from database (and cache)
				voUncached.push_back(rvoDBTypes[nIdx].getTypeOid());
			} // end if cache does not exist
			else
			{
				Debugger::logTrace(strMethodName, "Retrieving Type Oid " +
									Converter::longToString(rvoDBTypes[nIdx].getTypeOid()) + " from cache.");			
				// lets get the cached type
				rvoDBTypes[nIdx] = m_oDBTypeCache[rvoDBTypes[nIdx].getTypeOid()];
			} // end else cache exists
		} // end for more types in vector
		
		// only need to hit database, if something is not in the cache
		if (voUncached.size() > 0)
		{
			// sql statement used to retrieve all the view details
			string strSQL = "SELECT typname as typename, oid as typeoid "
											"FROM pg_type "
											"WHERE oid IN (";
			// lets build up IN clause
			for (int nIdx = 0; nIdx < voUncached.size(); nIdx++)
			{
				if (nIdx == 0)
				{
					strSQL += Converter::longToString(voUncached[nIdx]);
				} // end if first oid
				else
				{
					strSQL += ", " + Converter::longToString(voUncached[nIdx]);				
				} // end else not first oid
			} // end for more oids to add
			// close sql
			strSQL += ")";
			
			DBTypeSet oDBTypeSet;																		
			// execute query										
			m_poDBConn->executeQuery(strSQL, oDBTypeSet.m_oTypeList);
			
			// lets update the hash table
			while (oDBTypeSet.next())
			{
				DBType oDBType;
				oDBTypeSet.getDBType(oDBType);
				Debugger::logTrace(strMethodName, "Adding Type Oid " +
									Converter::longToString(oDBType.getTypeOid()) + " to cache.");							
				m_oDBTypeCache[oDBType.getTypeOid()] = oDBType;
			} // end while more types
			
			// lets now update our vector with types from database
			for (int nIdx = 0; nIdx < rvoDBTypes.size(); nIdx++)
			{
				// lets check to see if the type has been cached yet
				if (m_oDBTypeCache[rvoDBTypes[nIdx].getTypeOid()].getTypeOid()
						!= rvoDBTypes[nIdx].getTypeOid())
				{
					// this type oid does not exist in the database
					// just leave the type object as is (i.e. the type name will be an empty string)
					Debugger::logTrace(strMethodName, "The Type Oid " +
										Converter::longToString(rvoDBTypes[nIdx].getTypeOid()) + " does not exist in the database.");
					DBType oDBType;
					oDBType.setTypeOid(rvoDBTypes[nIdx].getTypeOid());
					m_oDBTypeCache[rvoDBTypes[nIdx].getTypeOid()] = oDBType;										
				} // end if cache does not exist
				else
				{
					Debugger::logTrace(strMethodName, "Retrieving Type Oid " +
										Converter::longToString(rvoDBTypes[nIdx].getTypeOid()) + " from cache.");			
					// lets get the cached type
					rvoDBTypes[nIdx] = m_oDBTypeCache[rvoDBTypes[nIdx].getTypeOid()];
				} // end else cache exists
			
			} // end for more types in vector
													
		} // end if need to hit database
		
		Debugger::exited(strMethodName);
		
	} // end retrieveListOfTables
	