/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbfunctionset.h"
#include "../dbconnection.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"
#include "../../utils/converter.h"
#include "../types/dbtypemanager.h"

	/**
 	 * Constructor 	
   */		
  DBFunctionSet::DBFunctionSet()
  {
  	// do nothing
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBFunctionSet::~DBFunctionSet()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBFunctionSet::isEmpty()
	{
		return m_oFunctionList.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBFunctionSet::getCurrentIndex()
	{
		return m_oFunctionList.getCurrentIndex();
	} // end getCurrentIndex
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBFunctionSet::next()
	{
		return m_oFunctionList.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBFunctionSet::previous()
	{
		return m_oFunctionList.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBFunctionSet::reset()
	{
		m_oFunctionList.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBFunctionSet::first() throw (NoRecordException)
	{
		return m_oFunctionList.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBFunctionSet::last() throw (NoRecordException)
	{
		return m_oFunctionList.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBFunctionSet::getRecordCount()
	{
		return m_oFunctionList.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current Function.
	 * @param	roDBFunction the a DBFunction object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBFunctionSet::getDBFunction(DBFunction & roDBFunction) throw (NoRecordException)
	{
		string strMethodName = "DBFunctionSet::getDBFunction";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException("Cannot retrieve DBFunction object, no records exist.", "DBFunctionSet", "getDBFunction");
		} // end if empty
		try
		{
			roDBFunction.setFunctionName(m_oFunctionList.getFieldValue("proname"));
			roDBFunction.setFunctionComment(m_oFunctionList.getFieldValue("comment"));			
			roDBFunction.setFunctionOwner(m_oFunctionList.getFieldValue("usename"));						
			roDBFunction.setLanguage(m_oFunctionList.getFieldValue("lanname"));						
			roDBFunction.setReturnType(m_vstrReturnTypes[getCurrentIndex()]);
			for (int nIdx = 0; nIdx < m_vvstrArgumentTypes[getCurrentIndex()].size(); nIdx++)
			{
				roDBFunction.addArgumentType(m_vvstrArgumentTypes[getCurrentIndex()][nIdx]);
			} // end for more arg types
			
		} // end try
		catch (VPException e)
		{
			Debugger::logException(strMethodName, e);
		} // end catch	
		
		Debugger::exited(strMethodName);
		
	} // end getDBFunction

