/* $PostgresPy: pl/src/plpy.c,v 1.33 2004/07/28 10:24:38 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *    The Python procedural language implementation.
 */
#include <setjmp.h>
#include <pgpy/pputils.h>

#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <commands/trigger.h>
#include <executor/executor.h>
#include <executor/spi.h>
#include <fmgr.h>
#include <miscadmin.h>
#include <nodes/execnodes.h>
#include <nodes/nodes.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/elog.h>
#include <utils/syscache.h>
#include <utils/rel.h>
#include <pgpy/pg.h>
#include <pgpy/PGExcept.h>

#include <Python.h>
#include <compile.h>
#include <frameobject.h>
#include <structmember.h>
#include <eval.h>
#include <marshal.h>
#include <pgpy/py.h>

#include "pl.h"

#include <pgpy/datum.h>
#include <pgpy/tupd.h>
#include <pgpy/tup.h>
#include <pgpy/type.h>
#include <pgpy/obj.h>
#include <pgpy/rel.h>
#include <pgpy/module.h>
#include <pgpy/utils.h>

#if (CATALOG_VERSION_NO < 200407000)
#define work_mem 1024
#endif

#define PLPY_IF			"postgres"
#define PLPY_FILENAME	"Postgres"
#define KnownModes		(SFRM_Materialize | SFRM_ValuePerCall)

#define PLPYARG_SELF	"self"
#define PLPYARG_ARGS	"args"
#define PLPYARG_KEYS	"kw"
#define PLPYARG_RSI	"RSI"

#ifndef Py_GENOBJECT_H
PyTypeObject *_pygentype;
#define PyGen_Type (*_pygentype)
#define GEN_FRAME(ob) ((PyFrameObject *) PyATTR(ob,"gi_frame"))
#define PyGen_Check(G) (PyObject_TypeCheck(G, _pygentype))
#else
#define GEN_FRAME(ob) (((PyGenObject *)ob)->gi_frame)
#endif

#define FRAME_LOCALS(ob) (PyATTR(((PyObj)ob),"f_locals"))
#define GEN_FLOCALS(ob) FRAME_LOCALS(GEN_FRAME(ob))

#define PyGen_CheckCode(G,C) \
	(PyGen_Check(G) && ((PyObj) (GEN_FRAME(G)->f_code)) == ((PyObj) C))

#define PyGen_CheckSelfRef(G) \
	(PyGen_Check(G) && G == PyMapITEM(GEN_FLOCALS(G),"self"))


const unsigned short	plpy_version	= 0;
const char *language_name				= "Python";

static PyObj __main__		= NULL; /* Python's __main__ dictionary */
static PyObj __plpy__		= NULL; /* Merge of __main__ dictionary */
static PyObj mod_pg			= NULL; /* Postgres module */
static PyObj fnExtras		= NULL; /* A list of persisting fn_extra */

/*
 * AtEOXact_DecrementExtras - Decrement fnExtras items > 1 refcount 
 *
 * Some tricky stuff happens here if a generator is being dealt with:
 * the generator owns a reference to itself; the self argument given
 * to the function is overwritten with a reference to the generator,
 * so as to provide the author with information
 */
static void
AtEOXact_DecrementExtras(bool isCommit, void *arg)
{
	PyObj fnExtra;
	
	while (PySequence_Size(fnExtras) > 0 &&
			(fnExtra = PyList_GET_ITEM(fnExtras, 0)))
	{
		if (PyGen_CheckSelfRef(fnExtra))
			PyMapping_DelItemString(GEN_FLOCALS(fnExtra), "self");

		PySequence_DelItem(fnExtras, 0);

		if (fnExtra->ob_refcnt > 0)
			DECREF(fnExtra);
	}

	UnregisterEOXactCallback(AtEOXact_DecrementExtras, NULL);
}

/*
 * fnExtras_Keep
 */
static void
fnExtras_Keep(PyObj fnextra)
{
	if (PySequence_Length(fnExtras) == 0)
		RegisterEOXactCallback(AtEOXact_DecrementExtras, NULL);

	PyList_Append(fnExtras, fnextra);
}

/*
 * fnExtras_Lose
 */
static void
fnExtras_Lose(PyObj fnextra)
{
	PySequence_DelItem(fnExtras, PySequence_Index(fnExtras, fnextra));

	if (PySequence_Length(fnExtras) == 0)
		UnregisterEOXactCallback(AtEOXact_DecrementExtras, NULL);
}

/*
 * fnExtras_Replace - Replace rather than Lose and Keep(avoid re-registration)
 */
static void
fnExtras_Replace(PyObj old, PyObj new)
{
	if (old)
		PyList_SetItem(fnExtras, PySequence_Index(fnExtras, old), new);
	else
		fnExtras_Keep(new);
}

static void
srflinfo_free(Datum arg)
{
	FmgrInfo *fli = (FmgrInfo *) DatumGetPointer(arg);
	PyObj fnextra;
	fnextra = (PyObj)fli->fn_extra;
	
	fnExtras_Lose(fnextra);
	DECREF(fnextra);

	fli->fn_extra = NULL;
}


void
lang_initialize(void)
{
	static PyObj __plpy__str = NULL;

	/* Only done once. Never Finalized. */
	if (!Py_IsInitialized())
	{
		Py_SetProgramName("Postgres");
		Py_Initialize();

		mod_pg = PyImport_ImportModule(PLPY_IF".pg"XSTR(PGV_MM));
		if (mod_pg == NULL)
			lereport("failed to import Postgres module",
				errhint("check and make sure that the module is installed"
					"in the Python site-packages that the PL is linked against")
			);

		fnExtras = PyList_New(0);

#ifdef PyGen_Type
		{
			PyObj tm;
			tm = PyImport_ImportModule("types");

			if (tm)
			{
				_pygentype = (PyTypeObject*)PyAttr(tm, "GeneratorType");
				DECREF(tm);
			}

			if (_pygentype == NULL)
				lereport("failed to fetch generator type from types module",
					errhint("Your Python installation is likely to be broken")
				);
		}
#endif
		if (__plpy__ == NULL)
		{
			__main__ = PyModule_GetDict(PyImport_AddModule("__main__")); 
			PyDict_SetItemString(__main__, "Postgres", mod_pg);
			__plpy__ = PyDict_New();
			PyDict_SetItemString(PyImport_GetModuleDict(), "__plpy__", __plpy__);
		}
		if (__plpy__str == NULL)
		{
			__plpy__str = PYSTR("__plpy__");
		}
	}

	/*
	 * Duplicate the imported __main__'s dictionary so not to potentially
	 * infect it with a broken state if/when an error occurs.
	 */

	if (PyDict_Update(__plpy__, __main__) < 0)
		lereport("failed to merge __main__ into __plpy__");

	PyDict_SetItemString(__plpy__, "__name__", __plpy__str);

	/* XXX: Do GUC initializations */
}

void
lang_finalize(void)
{
	PyDict_Clear(__plpy__);

#ifdef PY_FIXED_FINALIZE
	/* Only finalize under the fictional state that it is fixed. */
	Py_Finalize();
#endif
}

/*
 * indent
 *
 * Reallocate src with a volume of
 *		len(src) + fpad + lpad + (\n_count(src) * infact)
 * Start to copy src at msrc + fpad address
 * And set 'infact' number of characters following a \n to \t
 */
static char *
indent
(
	const char *const src,
	const unsigned short fpad,
	const unsigned short lpad,
	const unsigned short infact
)
{
	char *nsrc = NULL;
	register const char *frame = NULL;
	register char *msrc = NULL;
	register size_t sv, mv = 2;

	/*
	 * Existing Newline count, and source length.
	 */
	for (frame = src; *frame != '\0'; ++frame)
		if (*frame == '\n')
			mv+=infact;

	mv += (sv = frame - src);
	nsrc = msrc = palloc0(mv+fpad+lpad);

	msrc += fpad;

	/*
	 * Copy src to msrc with a new tab(s) following each newline.
	 */
	*msrc++ = '\t';
	for (frame = src; *frame != '\0'; ++frame)
	{
		*msrc = *frame;

		if (*(msrc++) == '\n')
			for (sv = infact; sv>0; --sv, ++msrc)
				*msrc = '\t';
	}

	return(nsrc);
}

#define FUNCNAME "f"
#define FUNCDEF "def " FUNCNAME "(" \
	PLPYARG_SELF ", " PLPYARG_ARGS ", " PLPYARG_KEYS ", " PLPYARG_RSI"):"
#define FUNCDEFLEN sizeof(FUNCDEF)

/*
 * plproc_FromSource - make a plproc from a procTuple and src(prosrc)
 */
plproc
plproc_FromSource(HeapTuple procTuple, char *src)
{
	Form_pg_proc procStruct = PROCSTRUCT(procTuple);
	PyCompilerFlags flags = {CO_OPTIMIZED};
	PyCodeObject *code = NULL;
	PyObj locals = NULL;
	PyObj func = NULL;
	char *msrc = NULL;

	msrc = indent(src, FUNCDEFLEN, 0, 1);
	Assert(msrc != NULL);

	memcpy(msrc, FUNCDEF, FUNCDEFLEN);
	msrc[FUNCDEFLEN-1] = '\n';
	{
		register Size msrcl = strlen(msrc)-1;
		if (msrc[msrcl] == '\t')
			msrc[msrcl] = '\n';
	}

	if ((locals = PyDict_New()))
		PyRun_StringFlags(msrc, Py_file_input, locals, locals, &flags);

	pfree(msrc);

	if (locals && !PyErr_Occurred())
	{
		func = PyFunction_GET_CODE(PyDict_GetItemString(locals, FUNCNAME));
		INCREF(func);
		code = (PyCodeObject*)func;

		DECREF(code->co_filename);
		code->co_filename = PyString_FromString(PLPY_FILENAME);

		DECREF(code->co_name);
		code->co_name = PyString_FromString(NameStr(procStruct->proname));

		DECREF(locals);
	}

	return(func);
}
#undef FUNCDEF
#undef FUNCNAME
#undef FUNCDEFLEN


/*
 * tbtostr - Python Traceback To String
 */
static char *
tbtostr(PyObj tb)
{
	PyObj mod_traceback, tb_format_tb;
	PyObj tbstrob = NULL, tbob;
	char *tbstr = NULL;

	Assert(tb && PyTraceBack_Check(tb));

	mod_traceback = PyImport_ImportModule("traceback");
	if (mod_traceback == NULL)
		goto err;

	tb_format_tb = PyAttr(mod_traceback, "format_tb");
	DECREF(mod_traceback);

	if (!tb_format_tb || !PyFunction_Check(tb_format_tb))
		goto err;

	tbob = PyObject_CallFunctionObjArgs(tb_format_tb, tb, NULL);
	DECREF(tb_format_tb);

	if (tbob)
	{
		PyObj iter, item, newline;

		iter = PyObject_GetIter(tbob);
		DECREF(tbob);

		tbstrob = PyString_FromString("");
		newline = PyString_FromString("\n");

		while ((item = PyIter_Next(iter)))
		{
			PyString_ConcatAndDel(&tbstrob,item);
			PyString_Concat(&tbstrob,newline);
		}
		DECREF(iter);
		DECREF(newline);

		if (tbstrob)
		{
			tbstr = pstrdup(PyString_AS_STRING(tbstrob));
			DECREF(tbstrob);
		}
	}
err:
	return(tbstr?tbstr:pstrdup("<Failed to render traceback>"));
}

void *
lang_error()
{
	return((void*) PyErr_Occurred());
}

/*
 * lang_errorstr - Get Language Error String
 *
 *	This function constructs an error string that correlates with
 * the current python exception.
 */
char *
lang_errorstr()
{
	char *vstr = "NULL Exception Instance",
		  *estr = "NULL Exception",
		  *tbstr = "\n";
	PyObj e = NULL, v = NULL, tb = NULL;
	PyObj epystr = NULL, vpystr = NULL;

	char *errstr = NULL, *perrstr = NULL;

	Assert( PyErr_Occurred() != NULL );

	PyErr_Fetch(&e, &v, &tb);
	PyErr_NormalizeException(&e, &v, &tb);

	if (e != NULL)
	{
		epystr = PyObject_Str(e);
		DECREF(e);
		if (epystr)
			estr = pstrdup(PyString_AS_STRING(epystr));
	}

	if (v != NULL)
	{
		vpystr = PyObject_Str(v);
		DECREF(v);
		if (vpystr)
			vstr = PyString_AS_STRING(vpystr);
	}

	if (tb != NULL)
	{
		tbstr = tbtostr(tb);
		DECREF(tb);
	}
	else
		tbstr = pstrdup("  <NULL Traceback>\n");

	asprintf(&errstr, "(most recent call last):\n%s%s %s", tbstr, estr, vstr);
	perrstr = pstrdup(errstr);
	free(errstr);
	pfree(tbstr);

	XDECREF(epystr);
	XDECREF(vpystr);

	return(perrstr);
}

char *
plproc_ToBinary(plproc code, Size *len)
{
	PyObj ob = (PyObj) code;
	Size size;
	char *data;
	PyObj marshald_ob;

	*len = 0;

	marshald_ob = PyMarshal_WriteObjectToString(ob);
	if (marshald_ob == NULL)
		return(NULL);

	size = (Size)PyString_Size(marshald_ob);
	if (size <= 2)
		size = 0;

	data = palloc(size);

	*len = size;
	memcpy(data, PYASSTR(marshald_ob), size);

	DECREF(marshald_ob);
	return(data);
}

plproc
plproc_FromBinary(char *from, Size len)
{
	PyObj reado;
	reado = PyMarshal_ReadObjectFromString(from, len);
	return((plproc)reado);
}

void
plproc_free(plproc proc)
{
	Assert(proc != NULL);
	DECREF((PyObj)proc);
}

static Tuplestorestate *
Materialize(PyObj seq, Oid typoid, TupleDesc *outDesc)
{
	PyObj iter, sob;
	Tuplestorestate *tss;
	HeapTuple ht;
	TupleDesc desc;

	iter = PyObject_GetIter(seq);
	if (iter == NULL)
		lereport(
			"failed to get sequence's iterator for materialization",
		);

	*outDesc = desc = TupleDesc_FromTypeOid(typoid);
	
	tss = tuplestore_begin_heap(true, true, work_mem);
	while ((sob = PyIter_Next(iter)))
	{
		if (PgTup_TypeCheck(sob) || PyTuple_Check(sob))
		{
			ht = HeapTuple_FromIterableAndTupleDesc(sob, desc);
		}
		else
		{
			char null = ' ';
			Datum single;

			if (desc->natts > 1)
			{
				DECREF(sob);
				DECREF(iter);
				FreeTupleDesc(desc);
				tuplestore_end(tss);

				ereport(ERROR,(
					errmsg("tuple descriptor for materialization requires more "
						"than one attribute; received an object that was "
						"not a tuple, or a Postgres.Tuple"),
					errdetail("XXX: good info about the invalid object")
				));
			}
			if (sob == Py_None || (PgObj_TypeCheck(sob) && PgObj_IsNull(sob)))
			{
				single = 0;
				null = 'n';
			}
			else if (PgObj_TypeCheck(sob) && PgObj_FetchTypeOid(sob) == typoid)
				single = PgDat_FetchDatum(sob);
			else
				single = Datum_FromPyObjectAndTypeOid(sob, typoid);
			
			ht = heap_formtuple(desc, &single, &null);
		}

		tuplestore_puttuple(tss, (void *) ht);
		heap_freetuple(ht);
		DECREF(sob);
	}
	DECREF(iter);
	
	return(tss);
}

static PyObj
PyObj_FromRSI(ReturnSetInfo *rsi)
{
	PyObj rsid, etd;
	rsid = PyDict_New();
	
	if (rsi->expectedDesc)
	{
		etd = PyPgTupleDesc_New(rsi->expectedDesc);
		PyDict_SetItemString(rsid, "expected", etd);
		DECREF(etd);
	}

	etd = PYINT(rsi->allowedModes);
	PyDict_SetItemString(rsid, "allowedModes", etd);
	DECREF(etd);

	return(rsid);
}

Datum
pl_main(PG_FUNCTION_ARGS)
{
	EXC_DECLARE();
	volatile MemoryContext fn_mcxt;
	volatile struct plcache *cache;
	Form_pg_proc procStruct;
	ReturnSetInfo *rsi;

	volatile TriggerData *td = NULL;
	volatile HeapTuple same_trigtup = NULL;

	volatile Datum rv = 0;

	volatile PyObj *extra;
	volatile PyObj code, rob, args, keys, iter, previter = NULL;
	volatile PyObj rsio = NULL;
	volatile PyObj gf_locals = NULL;
	int i;

	fn_mcxt = fcinfo->flinfo->fn_mcxt;
	cache = (struct plcache *)fcinfo->flinfo->fn_addr;
	Assert(cache != NULL && cache->proc);

	code = cache->code;
	Assert(code != NULL);

	procStruct = PROCSTRUCT(cache->proc);

	extra = (PyObj *) (&(fcinfo->flinfo->fn_extra));
	iter = *extra;

	rsi = (ReturnSetInfo *) fcinfo->resultinfo;

	if (fcinfo->context && nodeTag(fcinfo->context) == T_TriggerData)
	{
		td = (TriggerData *) fcinfo->context;
		iter = NULL;
	}
	else
		td = NULL;

	Assert(td && iter == NULL);
	
	if (iter)
	{
		if (PyGen_CheckCode(iter, code)) /* Continuing generator */
		{
			gf_locals = GEN_FLOCALS(iter);
			args = PyMapItem(gf_locals, PLPYARG_ARGS);
			keys = PyMapItem(gf_locals, PLPYARG_KEYS);
		}
		else
		{
			args = NULL;
			keys = NULL;
		}
	}
	else /* New function call */
	{
		args = PyList_New(PG_NARGS());
		keys = PyDict_New();

		if (rsi)
		{
			/* For sanity, check the allowedModes */
			if (!(rsi->allowedModes & KnownModes))
				ereport(ERROR, (
					errmsg("PL/Py only supports materialied or VPC SRFs"),
					errdetail("allowedModes was 0x%X, PL/Py is capable of 0x%X",
						rsi->allowedModes, KnownModes)
				));

			rsio = PyObj_FromRSI(rsi);
		}
	}


	/* Handle an ERROR */
	EXCEPT (
		XDECREF(iter);
		XDECREF(rob);
		XDECREF(args);
		XDECREF(keys);
		XDECREF(rsio);
	);


	/*
	 * Argument Initialization Section
	 */

	if (td)
	{
		HeapTuple ht;
		Trigger *tg = td->tg_trigger;
		PyObj tto, tdo, ob;

		ob = PYINT(td->tg_event);
		PyDict_SetItemString(keys, "EVENT", ob);
		DECREF(ob);

		ob = PYINT(tg->tgoid);
		PyDict_SetItemString(keys, "TRIGGER_OID", ob);
		DECREF(ob);

		ob = PYSTR(tg->tgname);
		PyDict_SetItemString(keys, "TRIGGER_NAME", ob);
		DECREF(ob);

		for (i = 0; i < tg->tgnargs; ++i)
		{
			ob = PYSTR(tg->tgargs[i]);
			PyList_SetItem(args, i, ob);
			DECREF(ob);
		}

		tdo = PgTupD_FromRelation(td->tg_relation);
		PyDict_SetItemString(keys, "TD", tdo);

		ht = td->tg_trigtuple;
		if (ht)
		{
			ht = heap_copytuple(ht);
			if (!td->tg_newtuple)
				same_trigtup = ht;

			tto = PyPgTuple_New(ht, tdo);
			PyDict_SetItemString(keys, "TUPLE", tto);
			DECREF(tto);
		}

		ht = td->tg_newtuple;
		if (ht)
		{
			ht = heap_copytuple(ht);
			same_trigtup = ht;

			tto = PyPgTuple_New(ht, tdo);
			PyDict_SetItemString(keys, "NEW", tto);
			DECREF(tto);
		}
		DECREF(tdo);
	}
	else if (args && keys) /* Normal Function */
	{
		PyObj arg;

		for (i = 0; i < PG_NARGS(); ++i)
		{
			if (PG_ARGISNULL(i))
				arg = Py_None;
			else
			{
				Datum od;
				HeapTuple tt;
				Form_pg_type ts;
				od = fcinfo->arg[i];
				tt = SearchSysCache(TYPEOID, procStruct->proargtypes[i], 0, 0, 0);
				ts = TYPESTRUCT(tt);

				/*
				 * No need to copy typbyval, special case for complex types
				 */
				
				if (ts->typtype == 'c')
				{
					TupleTableSlot *tts = (TupleTableSlot *) od;
					HeapTuple ht = heap_copytuple(tts->val);
					TupleDesc td =
						CreateTupleDescCopyConstr(tts->ttc_tupleDescriptor);
					PyObj tdo = PyPgTupleDesc_New(td);

					arg = PyPgTuple_New(ht, tdo);
					DECREF(tdo);

					if (tts->ttc_shouldFree)
						heap_freetuple(tts->val);
					if (tts->ttc_shouldFreeDesc)
						FreeTupleDesc(tts->ttc_tupleDescriptor);
				}
				else
				{
					if (!ts->typbyval)
						od = datumCopy(od, false, ts->typlen);
					arg = PyObj_FromDatumAndTypeTuple(od, tt);
				}

				ReleaseSysCache(tt);
			}

			/*
			 * Use PyList_SetItem, rather than Append
			 * to support _overwriting_ a generator's args
			 */
			PyList_SetItem(args, i, arg);

#ifdef Anum_pg_proc_proargnames
			if (cache->argnames[i] && strlen(cache->argnames[i]) > 0)
				PyDict_SetItemString(keys, cache->argnames[i], arg);
#endif
		}
	}


	/*
	 * Execution Section
	 */

	if (iter) /* Run the iterator */
	Iterate:
	{
		rob = PyIter_Next(iter);
		if (PyErr_Occurred())
			lereport("iterator raised exception on next");

		/* StopIteration */
		if (rob == NULL)
		{
			if (rsi)
			{
				rsi->isDone = ExprEndResult;
				UnregisterExprContextCallback(rsi->econtext,
						srflinfo_free, PointerGetDatum(fcinfo->flinfo));

				RETURN(0);
			}
			else /* Get next value for non-SRFs, if necessary */
			{
				if (previter) /* Only do this once. */
					ereport(ERROR,(
						errmsg("empty iterator encountered"),
						errhint(
							"make sure that the function's iterator returns None,"
							"instead of having an empty iterator"
						)
					));
				previter = iter;

				DECREF(iter);
				iter = NULL;
				*extra = NULL;
				goto Execute;
			}
		}

		if (gf_locals)
		{
			DECREF(args);
			args = NULL;
			DECREF(keys);
			keys = NULL;
		}
	}
	else /* Execute the function */
	Execute:
	{
		PyObj cargs[4] = {code, args, keys, rsio?rsio:Py_None};
		rob = PyEval_EvalCodeEx((PyCodeObject*) code, __plpy__, NULL, 
			cargs, 4, NULL, 0, NULL, 0, NULL);

		if (rob == NULL)
			lereport("function raised exception");

		DECREF(args);
		args = NULL;
		DECREF(keys);
		keys = NULL;
		XDECREF(rsio);
		rsio = NULL;

		/*
		 * self of a generator, should be the generator, not the function
		 * but ONLY IF it is a generator of the code object that was invoked
		 * This is done regardless of the function being an SRF.
		 */
		if (PyGen_CheckCode(rob, code))
			PyMapping_SetItemString(GEN_FLOCALS(rob), "self", rob);

		/* Select return mode, if returning a set */
		if (rsi)
		{
			/*
			 * I already validated that VPC or Mat was allowed in a sanity check
			 * at the beginning of the function, so unconditionally select Mat
			 * if VPC is not allowed. But if it is allowed, only select Mat if
			 * Mat is allowed AND the returned object is NOT an iterator.
			 *
			 * This allows the user to specify VPC in some circumstances.
			if (!(rsi->allowedModes & SFRM_ValuePerCall) ||
				(rsi->allowedModes & SFRM_Materialize &&
				 !PyIter_Check(rob)))
			 */

			/* XXX: Always materialize when allowed */
			if (rsi->allowedModes & SFRM_Materialize)
			{
				MemoryContext omc;
				rsi->returnMode = SFRM_Materialize;
				rsi->isDone = ExprSingleResult;

				omc = MemoryContextSwitchTo(rsi->econtext->ecxt_per_query_memory);
				rsi->setResult = Materialize(rob,
						procStruct->prorettype, &(rsi->setDesc));
				MemoryContextSwitchTo(omc);

				DECREF(rob);
				RETURN(0);
			}
			else
			{
				rsi->returnMode = SFRM_ValuePerCall;
				rsi->isDone = ExprMultipleResult;
				rsi->setDesc = rsi->expectedDesc;
				RegisterExprContextCallback(rsi->econtext,
						srflinfo_free, PointerGetDatum(fcinfo->flinfo));
				
				iter = PyObject_GetIter(rob);
				if (iter == NULL || PyErr_Occurred())
					lereport("failed to get iterator for VPC SRF");
				DECREF(rob);
				rob = NULL;
				
				*extra = iter;
				fnExtras_Keep(iter);
				goto Iterate;
			}
		}
		else if (PyIter_Check(rob))
		{
			*extra = rob;
			fnExtras_Replace(previter, rob);

			iter = rob;
			rob = NULL;
			goto Iterate;
		}
	}

	
	/*
	 * Return Section
	 *
	 * XXX: Fill in a summary of what this section
	 */

	if (rob == Py_None)
	{
		if (td == NULL)
			fcinfo->isnull = true;
	}
	else if (td) /* Qualify event with tuple */
	{
		HeapTuple rtup = NULL;
		MemoryContext oldcxt;

		if (TRIGGER_FIRED_AFTER(td->tg_event))
		{
			ereport(WARNING,(
				errmsg("ignoring return from trigger function fired after"),
				errdetail("function executed by trigger '%s' on '%s'",
					td->tg_trigger->tgname,
					RelationGetRelationName(td->tg_relation)),
				errhint("return None in the function to quiet this")
			));
		}

		oldcxt = MemoryContextSwitchTo(plFormerContext);
		if (PgTup_TypeCheck(rob))
		{
			rtup = PgTup_FetchHT(rob);
			if (rtup == same_trigtup)
				rtup =
					td->tg_newtuple ?
					td->tg_newtuple :
					td->tg_trigtuple;
			else
				rtup = heap_copytuple(rtup);
		}
		else if (PyIter_Check(rob))
		{
			rtup = HeapTuple_FromIterableAndTupleDesc(rob,
									RelationGetDescr(td->tg_relation));
		}
		else
			ereport(ERROR,(
				errmsg("erroneous return object"),
				errcontext("trigger procedure"),
				errdetail("expecting a Postgres.Tuple or a sequence, got %s",
					rob->ob_type->tp_name)
			));
		MemoryContextSwitchTo(oldcxt);	
		rv = PointerGetDatum(rtup);
	}
	else if (PgObj_TypeCheck(rob))
	{
		PgObj Rob = (PgObj)rob;
		Oid roid;
		Form_pg_type typs;

		roid = PgObj_FetchTypeOid(Rob);
		typs = PGOTYPESTRUCT(Rob);

		if (roid != procStruct->prorettype)
			ereport(ERROR,(
				errmsg("returned type does not match function's"),
				errdetail("expecting %d, got %d", procStruct->prorettype, roid)
			));

		if (PgObj_IsNull(Rob))
		{
			fcinfo->isnull = true;
			rv = 0;
		}
		else
		{
			fcinfo->isnull = false;
			rv = Rob->ob_datum;

			if (!typs->typbyval)
			{
				MemoryContext omc;
				omc = MemoryContextSwitchTo(fn_mcxt);
				rv = datumCopy(Rob->ob_datum, false, typs->typlen);
				MemoryContextSwitchTo(omc);
			}
		}
	}
	else if (PgTup_TypeCheck(rob))
	{
		MemoryContext oldcxt;
		HeapTuple rettype;
		Form_pg_type tstruct;
		char typtype;

		rettype = SearchSysCache(TYPEOID, procStruct->prorettype, 0, 0, 0);
		tstruct = TYPESTRUCT(rettype);
		
		typtype = tstruct->typtype;
		ReleaseSysCache(rettype);
		if (typtype != 'c')
			ereport(ERROR,(
				errmsg("function returned a Postgres.Tuple, "
					"but function return type is not a complex type")
			));

		oldcxt = MemoryContextSwitchTo(fn_mcxt);
		rv = PointerGetDatum(heap_copytuple(PgTup_FetchHT(rob)));
		MemoryContextSwitchTo(oldcxt);
	}
	else if (PgDat_TypeCheckExact(rob))
	{
		rv = PgDat_FetchDatum(rob);
	}
	else
		rv = Datum_FromPyObjectAndTypeOid(rob, procStruct->prorettype);

	DECREF(rob);
	RETURN(rv);
}
