/* Module:          SQLColAttributes.c
 *
 * Description:     Has been replaced by SQLColAttribute. 
 *
 * Classes:         
 *
 * API functions:   SQLColAttributes
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLColAttributes(
								   SQLHSTMT			hDrvStmt,
								   SQLUSMALLINT		nCol,
								   SQLUSMALLINT		nDescType,
								   SQLPOINTER		pszDesc,
								   SQLSMALLINT		nDescMax,
								   SQLSMALLINT		*pcbDesc,
								   SQLINTEGER		*pfDesc
								   )
{
	static char *func = "SQLColAttributes";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	Int4 field_type = 0;
	ConnInfo *ci;
	int unknown_sizes;
	int cols = 0;
	char parse_ok;
	SQLRETURN result;
	char *p = NULL;
	int len = 0, value = 0;

	mylog("%s: entering...\n", func);

	if( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	ci = &(stmt->hdbc->connInfo);

	/*	Dont check for bookmark column.  This is the responsibility
		of the driver manager.  For certain types of arguments, the column
		number is ignored anyway, so it may be 0.
	*/

	nCol--;

	unknown_sizes = globals.unknown_sizes;          /* atoi(ci->unknown_sizes); */
	if (unknown_sizes == UNKNOWNS_AS_DONTKNOW)
	{
		/* not appropriate for SQLColAttributes() */
		unknown_sizes = UNKNOWNS_AS_MAX;
	}

	parse_ok = FALSE;
	if (globals.parse && stmt->statement_type == STMT_TYPE_SELECT) 
	{

		if (stmt->parse_status == STMT_PARSE_NONE)
		{
			mylog("SQLColAttributes: calling parse_statement\n");
			parse_statement(stmt);
		}

		cols = stmt->nfld;

		/*	Column Count is a special case.  The Column number is ignored
			in this case.
		*/
		if (nDescType == SQL_COLUMN_COUNT)
		{
			if (pfDesc)
				*pfDesc = cols;

			return SQL_SUCCESS;
		}

		if (stmt->parse_status != STMT_PARSE_FATAL && stmt->fi && stmt->fi[nCol])
		{

			if (nCol >= cols)
			{
				stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
				stmt->errormsg = "Invalid column number in DescribeCol.";
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}

			field_type = stmt->fi[nCol]->type;
			if (field_type > 0)
			{
				parse_ok = TRUE;
			}
		}
	}

	if ( ! parse_ok) 
	{
		SC_pre_execute(stmt);       

		mylog("**** SQLColAtt: result = %u, status = %d, numcols = %d\n", stmt->result, stmt->status, stmt->result != NULL ? QR_NumResultCols(stmt->result) : -1);

		if ( (NULL == stmt->result) || ((stmt->status != STMT_FINISHED) && (stmt->status != STMT_PREMATURE)) ) 
		{
			stmt->errormsg = "Can't get column attributes: no result found.";
			stmt->errornumber = STMT_SEQUENCE_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		cols = QR_NumResultCols(stmt->result);

		/*	Column Count is a special case.  The Column number is ignored
			in this case.
		*/
		if (nDescType == SQL_COLUMN_COUNT)
		{
			if (pfDesc)
				*pfDesc = cols;

			return SQL_SUCCESS;
		}

		if (nCol >= cols)
		{
			stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
			stmt->errormsg = "Invalid column number in DescribeCol.";
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		field_type = QR_get_field_type(stmt->result, nCol);
	}

	mylog("colAttr: col %d field_type = %d\n", nCol, field_type);

	switch(nDescType)
	{
	case SQL_COLUMN_AUTO_INCREMENT:
		value  = pgtype_auto_increment(stmt, field_type);
		if (value == -1)
		{
			/*  non-numeric becomes FALSE (ODBC Doc) */
			value = FALSE;
		}
				
		break;

	case SQL_COLUMN_CASE_SENSITIVE:
		value = pgtype_case_sensitive(stmt, field_type);
		break;

	/* 	This special case is handled above.

	case SQL_COLUMN_COUNT: 
	*/

    case SQL_COLUMN_DISPLAY_SIZE:
		value = (parse_ok) ? stmt->fi[nCol]->display_size : pgtype_display_size(stmt, field_type, nCol, unknown_sizes);

		mylog("SQLColAttributes: col %d, display_size= %d\n", nCol, value);

        break;

	case SQL_COLUMN_LABEL:
		if (parse_ok && stmt->fi[nCol]->alias[0] != '\0') {
			p = stmt->fi[nCol]->alias;

			mylog("SQLColAttr: COLUMN_LABEL = '%s'\n", p);
			break;

		}	/* otherwise same as column name -- FALL THROUGH!!! */

	case SQL_COLUMN_NAME:

		p = (parse_ok) ? stmt->fi[nCol]->name : QR_get_fieldname(stmt->result, nCol);

		mylog("SQLColAttr: COLUMN_NAME = '%s'\n", p);
		break;

	case SQL_COLUMN_LENGTH:
		value = (parse_ok) ? stmt->fi[nCol]->length :  pgtype_length(stmt, field_type, nCol, unknown_sizes); 

		mylog("SQLColAttributes: col %d, length = %d\n", nCol, value);
        break;

	case SQL_COLUMN_MONEY:
		value = pgtype_money(stmt, field_type);
		break;

	case SQL_COLUMN_NULLABLE:
		value = (parse_ok) ? stmt->fi[nCol]->nullable : pgtype_nullable(stmt, field_type);
		break;

	case SQL_COLUMN_OWNER_NAME:
		p = "";
		break;

	case SQL_COLUMN_PRECISION:
		value = (parse_ok) ? stmt->fi[nCol]->precision : pgtype_precision(stmt, field_type, nCol, unknown_sizes);

		mylog("SQLColAttributes: col %d, precision = %d\n", nCol, value);
        break;

	case SQL_COLUMN_QUALIFIER_NAME:
		p = "";
		break;

	case SQL_COLUMN_SCALE:
		value = pgtype_scale(stmt, field_type, nCol);
		break;

	case SQL_COLUMN_SEARCHABLE:
		value = pgtype_searchable(stmt, field_type);
		break;

    case SQL_COLUMN_TABLE_NAME:

		p = (parse_ok && stmt->fi[nCol]->ti) ? stmt->fi[nCol]->ti->name : "";

		mylog("SQLColAttr: TABLE_NAME = '%s'\n", p);
        break;

	case SQL_COLUMN_TYPE:
		value = pgtype_to_sqltype(stmt, field_type);
		break;

	case SQL_COLUMN_TYPE_NAME:
		p = pgtype_to_name(stmt, field_type);
		break;

	case SQL_COLUMN_UNSIGNED:
		value = pgtype_unsigned(stmt, field_type);
		if(value == -1)
		{
			/* non-numeric becomes TRUE (ODBC Doc) */
			value = TRUE;
		}

		break;

	case SQL_COLUMN_UPDATABLE:
		/*	Neither Access or Borland care about this.

		if (field_type == PG_TYPE_OID)
			*pfDesc = SQL_ATTR_READONLY;
		else
		*/

		value = SQL_ATTR_WRITE;

		mylog("SQLColAttr: UPDATEABLE = %d\n", value);
		break;
    }

	result = SQL_SUCCESS;

	if (p) 
	{  /* char/binary data */
		len = strlen(p);

		if (pszDesc)
		{
			strncpy_null((char *)pszDesc, p, (size_t)nDescMax);

			if (len >= nDescMax) 
			{
				result = SQL_SUCCESS_WITH_INFO;
				stmt->errornumber = STMT_TRUNCATED;
				stmt->errormsg = "The buffer was too small for the result.";
			}
		}

		if (pcbDesc) 
		{
			*pcbDesc = len;
		}
	}
	else 
	{	/* numeric data */

		if (pfDesc)
		{
			*pfDesc = value;
		}

	}


    return result;
}

