/* Module:          SQLDriverConnect.c
 *
 * Description:     An alternative to SQLConnect.
 *
 * Classes:         
 *
 * API functions:   SQLDriverConnect
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

void dconn_get_connect_attributes(UCHAR FAR *connect_string, ConnInfo *ci);

#ifdef WIN32
BOOL FAR PASCAL dconn_FDriverConnectProc(HWND hdlg, UINT wMsg, WPARAM wParam, LPARAM lParam);
RETCODE dconn_DoDialog(HWND hwnd, ConnInfo *ci);

extern HINSTANCE NEAR s_hModule;               /* Saved module handle. */
#endif


SQLRETURN SQL_API SQLDriverConnect(
								   SQLHDBC      hDrvDbc,
								   SQLHWND      hWnd,
								   SQLCHAR *szConnStrIn,
								   SQLSMALLINT     nConnStrIn,
								   SQLCHAR *szConnStrOut,
								   SQLSMALLINT     cbConnStrOutMax,
								   SQLSMALLINT  *pnConnStrOut,
								   SQLUSMALLINT     nDriverCompletion
								   )
{
	static char *func = "SQLDriverConnect";
	ConnectionClass *conn = (ConnectionClass *) hDrvDbc;
	ConnInfo *ci;

	#ifdef WIN32
	SQLRETURN dialog_result;
	#endif

	SQLRETURN result;
	char connStrIn[MAX_CONNECT_STRING];
	char connStrOut[MAX_CONNECT_STRING];
	int retval;
	char password_required = FALSE;
	int len = 0;


	mylog("%s: entering...\n", func);

	if ( ! conn)
	{
		CC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	make_string(szConnStrIn, nConnStrIn, connStrIn);

	mylog("**** SQLDriverConnect: nDriverCompletion=%d, connStrIn='%s'\n", nDriverCompletion, connStrIn);
	qlog("conn=%u, SQLDriverConnect( in)='%s', nDriverCompletion=%d\n", conn, connStrIn, nDriverCompletion);

	ci = &(conn->connInfo);

	/*	Parse the connect string and fill in conninfo for this hDrvDbc. */
	dconn_get_connect_attributes(connStrIn, ci);

	/*	If the ConnInfo in the hDrvDbc is missing anything, */
	/*	this function will fill them in from the registry (assuming */
	/*	of course there is a DSN given -- if not, it does nothing!) */
	getDSNinfo(ci, CONN_DONT_OVERWRITE);

	/*	Fill in any default parameters if they are not there. */
	getDSNdefaults(ci);
	/*	initialize pg_version */
	CC_initialize_pg_version(conn);

	#ifdef WIN32
	dialog:
	#endif
	ci->focus_password = password_required;

	switch(nDriverCompletion)
	{
		#ifdef WIN32
		case SQL_DRIVER_PROMPT:
			dialog_result = dconn_DoDialog(hWnd, ci);
			
			if(dialog_result != SQL_SUCCESS)
			{
				return dialog_result;
			}
			break;

		case SQL_DRIVER_COMPLETE_REQUIRED:
			/*	Fall through */

		case SQL_DRIVER_COMPLETE:
			/* Password is not a required parameter. */
			if( ci->username[0] == '\0' || 
				ci->server[0] == '\0' || 
				ci->database[0] == '\0' ||
				ci->port[0] == '\0' ||
				password_required)
			{
				dialog_result = dconn_DoDialog(hWnd, ci);
				if(dialog_result != SQL_SUCCESS)
				{
					return dialog_result;
				}
			}
			break;

		#else
		case SQL_DRIVER_PROMPT:
		case SQL_DRIVER_COMPLETE:
		case SQL_DRIVER_COMPLETE_REQUIRED:
		#endif

		case SQL_DRIVER_NOPROMPT:
			break;
	}
	

	/*	Password is not a required parameter unless authentication asks for it.
		For now, I think it's better to just let the application ask over and over until
		a password is entered (the user can always hit Cancel to get out)
	*/
	if( ci->username[0] == '\0' ||
		ci->server[0] == '\0' ||
		ci->database[0] == '\0' || 
		ci->port[0] == '\0')
	{
		/*		(password_required && ci->password[0] == '\0')) */
		return SQL_NO_DATA_FOUND;
	}


	/* do the actual connect */
	retval = CC_connect(conn, password_required);
	if (retval < 0)
	{
		/* need a password */
		if (nDriverCompletion == SQL_DRIVER_NOPROMPT)
		{
			CC_log_error(func, "Need password but Driver_NoPrompt", conn);
			return SQL_ERROR;	/* need a password but not allowed to prompt so error */
		}
		else
		{
			#ifdef WIN32
			password_required = TRUE;
			goto dialog;
			#else
			return SQL_ERROR;	/* until a better solution is found. */
			#endif
		}
	}
	
	else if (retval == 0)
	{
		/*	error msg filled in above */
		CC_log_error(func, "Error from CC_Connect", conn);
		return SQL_ERROR;
	}

	/*********************************************/
	/*     Create the Output Connection String   */
	/*********************************************/
	result = SQL_SUCCESS;

	makeConnectString(connStrOut, ci);
	len = strlen(connStrOut);

	if(szConnStrOut)
	{
		/*	Return the completed string to the caller. The correct method is to 
			only construct the connect string if a dialog was put up, otherwise, 
			it should just copy the connection input string to the output.  
			However, it seems ok to just always	construct an output string.  There
			are possible bad side effects on working applications (Access) by 
			implementing the correct behavior, anyway. 
		*/
		strncpy_null(szConnStrOut, connStrOut, cbConnStrOutMax);

		if (len >= cbConnStrOutMax)
		{
			result = SQL_SUCCESS_WITH_INFO;
			conn->errornumber = CONN_TRUNCATED;
			conn->errormsg = "The buffer was too small for the result.";
		}
	}
	
	if(pnConnStrOut)
	{
		*pnConnStrOut = len;
	}
	
	mylog("szConnStrOut = '%s'\n", szConnStrOut);
	qlog("conn=%u, SQLDriverConnect(out)='%s'\n", conn, szConnStrOut);


	mylog("SQLDRiverConnect: returning %d\n", result);
	return result;
}
