/* Module:          SQLParamData.c
 *
 * Description:     Used in conjunction with SQLPutData to supply parameter 
 *					data at statement execution time.
 *
 * Classes:         
 *
 * API functions:   SQLParamData
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLParamData(
							   SQLHSTMT		hDrvStmt,
							   SQLPOINTER	*pValue
							   )
{
	static char *func = "SQLParamData";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	int i, retval;

	mylog( "%s: entering...\n", func);

	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	mylog("%s: data_at_exec=%d, params_alloc=%d\n", func, stmt->data_at_exec, stmt->parameters_allocated);

	if (stmt->data_at_exec < 0)
	{
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		stmt->errormsg = "No execution-time parameters for this statement";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	if (stmt->data_at_exec > stmt->parameters_allocated)
	{
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		stmt->errormsg = "Too many execution-time parameters were present";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	/* close the large object */
	if ( stmt->lobj_fd >= 0)
	{
		lo_close(stmt->hdbc, stmt->lobj_fd);

		/* commit transaction if needed */
		if (!globals.use_declarefetch && CC_is_in_autocommit(stmt->hdbc))
		{
			QResultClass *res;
			char ok;

			res = CC_send_query(stmt->hdbc, "COMMIT", NULL);
			if (!res)
			{
				stmt->errormsg = "Could not commit (in-line) a transaction";
				stmt->errornumber = STMT_EXEC_ERROR;
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}
			ok = QR_command_successful(res);
			QR_Destructor(res);
			if (!ok)
			{
				stmt->errormsg = "Could not commit (in-line) a transaction";
				stmt->errornumber = STMT_EXEC_ERROR;
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}

			CC_set_no_trans(stmt->hdbc);
		}

		stmt->lobj_fd = -1;
	}


	/*	Done, now copy the params and then execute the statement */
	if (stmt->data_at_exec == 0)
	{
		retval = copy_statement_with_parameters(stmt);
		if (retval != SQL_SUCCESS)
		{
			return retval;
		}

		stmt->current_exec_param = -1;

		return SC_execute(stmt);
	}

	/*	Set beginning param;  if first time SQLParamData is called , start at 0.
		Otherwise, start at the last parameter + 1.
	*/
	i = stmt->current_exec_param >= 0 ? stmt->current_exec_param+1 : 0;

	/*	At least 1 data at execution parameter, so Fill in the token value */
	for ( ; i < stmt->parameters_allocated; i++)
	{
		if (stmt->parameters[i].data_at_exec == TRUE)
		{
			stmt->data_at_exec--;
			stmt->current_exec_param = i;
			stmt->put_data = FALSE;
			*pValue = stmt->parameters[i].buffer;	/* token */
			break;
		}
	}

	return SQL_NEED_DATA;
}
