/**
 * 
 */
package com.enterprisedb.dashboard.agent;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.logging.*;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;


/**
 * This class provides several utility methods to deal with Sources to monitor.
 * 
 * @author Usman
 * @version 0.8b
 */
public class DashboardAgent {

	private static ArrayList sourcesList = new ArrayList();

	private static Set invalidSources = new HashSet();
	private static Set invalidDBAUserSources = new HashSet();
	private static final Logger logger = Logger.getLogger(DashboardAgent.class
			.getName());

	private static boolean loaded = false;

	private static long confLastModified = 0;
	
	private static Source mgmtServer = new Source();

	/**
	 * Returns an array of sources whose users are not super user.
	 * 
	 * @return String[] Returns an array of sources which are invalid
	 * @author usman
	 */
	public synchronized static String[] getInvalidDBAUserSources() {
		return (String[]) invalidDBAUserSources
				.toArray(new String[invalidDBAUserSources.size()]);
	}
	/**
	 * Returns an array of sources which are invalid.
	 * 
	 * @return String[] Returns an array of sources which are invalid
	 * @author usman
	 */
	public synchronized static String[] getInvalidSources() {
		return (String[]) invalidSources.toArray(new String[invalidSources
				.size()]);
	}

	
	/**
	 * Checks for source database availability. Also checks if user provided in
	 * configure.xml file to connect to this databae is a DBA superuser or not.
	 * 
	 * @param src
	 *            Source as a string in form of HOST:PORT/DATABASE
	 * @throws DashboardAgentException
	 */
	public synchronized static void checkSourceConnection(String src)
			throws DashboardAgentException {
		checkSourceConnection(getSource(src));
	}
	
	/**
	 * Get Logs Displayable for Logs JSP page. 
	 * @return String[] of logs in form of host:port"
	 */
	public synchronized static String[]getLogDisplaySources() {
	    Set displayHosts = new HashSet();
	    for(int i=0; i<sourcesList.size();i++) {
	    	Source src = (Source)sourcesList.get(i);
	    	String display = src.getHost()+":"+src.getPort();
	    	displayHosts.add(display);
	    }
	    return (String[]) displayHosts.toArray(new String[displayHosts
	                                        				.size()]);
		
	}
	
	/**
	 * Gets a first database source which  matches with input parameter in 
	 * form of host:port
	 * @param source host:port
	 * @return host:port/database, null if not found
	 */
	public synchronized static String getLogFullSource(String source) {
		for(int i=0; i<sourcesList.size();i++) {
	    	Source src = (Source)sourcesList.get(i);
	    	if(source.equalsIgnoreCase(src.getHost()+":"+src.getPort())) {
	    		return src.getHost()+":"+src.getPort() + "/" + src.getDatabase();
	    	}
	    }
		return null;
	}

	/**
	 * Checks for Super user
	 * 
	 * @param con
	 *            Connection object
	 * @param user
	 *            The user to check in database
	 * @return true if super user, false otherwise
	 */
	public static synchronized boolean isSuperUser(Source source) {
		boolean superUser = false;
		Connection con = null;
		try {
			con = source.getConnection();
			// determine if we are a super user or not
			Statement st = con.createStatement();
			ResultSet rs = st
					.executeQuery("select usesuper from pg_user where usename = '"
							+ source.getUser() + "'");
			if (rs.next()) {
				superUser = rs.getBoolean(1);
			} else {
				// this is never suppose to happen - unless user is deleted from
				// db but exist in our configuration file
				logger.severe("No result from pg_user for user:"
						+ source.getUser());
			}
			rs.close();
			st.close();
		} catch (SQLException e) {
			logger.log(Level.SEVERE, e.getMessage(), e);
		} finally {

			if (con != null) {
				try {
					con.close();
				} catch (SQLException e) {

				}
			}
		}
		return superUser;
	}

	/**
	 * Checks for source database availability.
	 * 
	 * @param src
	 *            The source database as registered via loadConfig call
	 * @throws DashboardException
	 *             In case of source database is not available...
	 * @author usman
	 */
	public synchronized static void checkSourceConnection(Source source)
			throws DashboardAgentException {

		if (source != null) {
			Connection con = null;
			try {
				con = source.getConnection();
			} catch (SQLException e) {
				throw new DashboardAgentException(e);
			} finally {

				if (con != null) {
					try {
						con.close();
					} catch (SQLException e) {
						throw new DashboardAgentException(e);
					}
				}
			}
		} else {
			throw new DashboardAgentException("Invalid source parameter");
		}
	}

	/**
	 * Returns source instance based on passed argument
	 * 
	 * @param source
	 * @return Source source instance, null if not found
	 */
	public synchronized static Source getSource(String source) {
		Source _source = new Source();
		_source.setName(source);
		int index = sourcesList.indexOf(_source);
		if (index >= 0) {
			return (Source) sourcesList.get(index);
		}
		return null;
	}

	/**
	 * Returns iterator of Sources
	 * 
	 * @return
	 */
	public synchronized static Iterator getSourceIterator() {
		return sourcesList.iterator();
	}
	
	
	/**
	 * Load Configuration from XML File
	 * 
	 * @param file
	 * The XML file to load configuration from
	 * @throws DocumentException
	 * @version 0.8
	 */
	public synchronized static void loadConfig(File file)
			throws DashboardAgentException {
		// we want to load if configuration file is modified
		if (file.lastModified() != confLastModified) {
			confLastModified = file.lastModified();
			loaded = false;
			logger.info("configuration file modified...");
		}

		if (loaded) {
			// we are already loaded!
			return;
		}

		// read sources...
		SAXReader reader = new SAXReader();
		Document document;
		try {
			document = reader.read(file);
			// Populate all the source database with their respective
			// information from the configuration file
			// FIX (Uzi): Removed <name> and <url> and introduced <host> <port>
			// <database>
			// which will create URL in form of jdbc:edb://host:port/database
			//FIX 0.8: Invalid sources are seprated out
			List sourceNodesList = document.selectNodes("//config/source");
			synchronized (sourcesList) {
				sourcesList.clear();
				invalidDBAUserSources.clear();
				invalidSources.clear();
				
				for (Iterator nodes = sourceNodesList.iterator(); nodes
						.hasNext();) {
					Node source = (Node) nodes.next();
					Source sourceDB = new Source();
					String URL = "jdbc:edb://";

					sourceDB.setHost(source.selectSingleNode("host").getText());
					sourceDB.setPort(source.selectSingleNode("port").getText());
					sourceDB.setDatabase(source.selectSingleNode("database")
							.getText());
					sourceDB.setUser(source.selectSingleNode("user").getText());
					sourceDB.setPass(source.selectSingleNode("password")
							.getText());
					sourceDB.setReadOnly(Boolean.valueOf(
							source.selectSingleNode("readOnly").getText())
							.booleanValue());
					// Fix 0.7c (Uzi) This should be in form of
					// HOST:PORT/DATABASE
					sourceDB
							.setName(sourceDB.getHost() + ":"
									+ sourceDB.getPort() + "/"
									+ sourceDB.getDatabase());
					sourceDB
							.setUrl(URL + sourceDB.getHost() + ":"
									+ sourceDB.getPort() + "/"
									+ sourceDB.getDatabase());

					// check validity of source...
					try {
						checkSourceConnection(sourceDB);
						// check for super user
						if (isSuperUser(sourceDB)) {
							// Only add source if we do not already contains it
							if (!DashboardAgent.sourcesList.contains(sourceDB)) {
								DashboardAgent.sourcesList.add(sourceDB);
								// TODO:remove them from Invalid lists if they exist?

							}
						} else {
							invalidDBAUserSources.add(sourceDB.getName());
						}
					} catch (DashboardAgentException dae) {
						invalidSources.add(sourceDB.getName());
					}

				}
			}
			loaded = true;
		} catch (DocumentException e) {
			throw new DashboardAgentException(e);
		}

	}

	/**
	 * Obtains connection from a source
	 * 
	 * @param source
	 *            String
	 * @return Connection c
	 * @throws DashboardAgentException
	 * @author Usman
	 */
	public static synchronized Connection getSourceConnection(String src)
			throws DashboardAgentException {
		Source source = getSource(src);
		if (source != null) {
			try {
				return source.getConnection();
			} catch (SQLException e) {
				throw new DashboardAgentException(e);
			}
		} else {
			throw new DashboardAgentException("Invalid source parameter");
		}
	}

}