package com.enterprisedb.dashboard.agent;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Iterator;

import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRField;

/**
 * Custom JR DataSource for landing page of EnterpriseDB DBA Dashboard. Each row
 * will represent individual source
 * 
 * @author usman
 * @version 0.8
 */
public class SourceStatsJRDataSource implements JRDataSource {

	private static final String SOURCE_QUERY = "select pg_database.oid AS dboid,substr(version(),0,strpos(version(),'on')-1) as version, pg_get_userbyid(pg_database.datdba) as owner, "
			+ " pg_database.encoding, pg_stat_database.numbackends, (case when current_setting('log_min_duration_statement') = -1 then 'OFF' else 'ON' end) as qlogging,"
			+ " pg_stat_database.xact_commit, pg_stat_database.xact_rollback ,pg_stat_database.blks_read,"
			+ " pg_stat_database.blks_hit ,current_setting('stats_start_collector') as stats_start_collector,"
			+ " current_setting('stats_command_string') as stats_command_string,"
			+ " current_setting('stats_block_level') as stats_block_level, current_setting('stats_row_level') as stats_row_level,"
			+ " current_setting('stats_reset_on_server_start') as stats_reset_on_server_start, pg_tablespace.spcname "
			+ " from pg_database , pg_stat_database, pg_tablespace where pg_database.datname = pg_stat_database.datname"
			+ " and pg_database.dattablespace = pg_tablespace.oid"
			+ " and UPPER(pg_database.datname) = ?";

	private static final String SESSION_QUERY = "SELECT count(*) AS session_count FROM pg_stat_activity WHERE UPPER(datname) =?";
	private static final String SIZE_EXIST_QUERY = "SELECT count(proname) AS procCount FROM pg_proc WHERE UPPER(proname) = 'PG_LS_DIR'";
	private static final String DBSIZE_QUERY = "SELECT pg_size_pretty(pg_database_size(pg_database.oid)) AS dbsize"
			+ " FROM pg_stat_database,pg_database "
			+ " WHERE pg_database.oid =?";
	private static final String SERVER_UPTIME_QUERY = "SELECT modification FROM pg_stat_file('postmaster.pid')";
	
	// report fields...
	private ArrayList database_name = new ArrayList();
	private ArrayList version = new ArrayList();
	private ArrayList owner = new ArrayList();
	private ArrayList encoding = new ArrayList();
	private ArrayList numbackends = new ArrayList();
	private ArrayList xact_commit = new ArrayList();
	private ArrayList xact_rollback = new ArrayList();
	private ArrayList blks_read = new ArrayList();
	private ArrayList blks_hit = new ArrayList();
	private ArrayList dbSize = new ArrayList();
	private ArrayList session_count = new ArrayList();
	private ArrayList serverStartTime = new ArrayList();
        private ArrayList qlogging = new ArrayList();
        private ArrayList stats_block_level = new ArrayList();
	// default count = 0 means empty
	private int count = 0;

	/**
	 * Default constructor - initialize all fields
	 * 
	 * @param source
	 *            Iterator containing Source
	 */
	public SourceStatsJRDataSource(Iterator source) {
		if (source != null) {
			while (source.hasNext()) {
				Object o = source.next();
				// although we are 100% sure that this is a Source iterator, but
				// nevertheless...
				if (o instanceof Source) {
					Source src = (Source) o;
					this.database_name.add(src.getHost() + ":" + src.getPort()
							+ "/" + src.getDatabase());
					Connection con = null;
					try {
						con = src.getConnection();

						// Step 1: Major Data...
						PreparedStatement sourceStatement = con
								.prepareStatement(SOURCE_QUERY);

						sourceStatement.setString(1, src.getDatabase()
								.toUpperCase());
						ResultSet rsSource = sourceStatement.executeQuery();
						long dboid = 0;
						if (rsSource.next()) {
							dboid = rsSource.getInt("dboid");
							this.version.add(rsSource.getString("version"));
							this.owner.add(rsSource.getString("owner"));
							this.encoding.add(getEncoding(rsSource
									.getInt("encoding")));
							this.numbackends.add(new Long(rsSource
									.getLong("numbackends")));
							this.xact_commit.add(new Long(rsSource
									.getLong("xact_commit")));
							this.xact_rollback.add(new Long(rsSource
									.getLong("xact_rollback")));
							this.blks_read.add(new Long(rsSource
									.getLong("blks_read")));
							this.blks_hit.add(new Long(rsSource
									.getLong("blks_hit")));
                                                        this.qlogging.add(rsSource
                                                                        .getString("qlogging"));
                                                        this.stats_block_level.add(rsSource
                                                                        .getString("stats_block_level"));
						}
						rsSource.close();
						sourceStatement.close();
						// step 2: Session Count:
						PreparedStatement sessionStatement = con
								.prepareStatement(SESSION_QUERY);

						sessionStatement.setString(1, src.getDatabase()
								.toUpperCase());
						ResultSet rsSession = sessionStatement.executeQuery();
						if (rsSession.next()) {
							this.session_count.add(new Long(rsSession
									.getLong("session_count")));
						}
						rsSession.close();
						sessionStatement.close();

						// Step3: Determine dbsize
						// Step 3.1 Determine if PG_LS_DIR function exists
						Statement stSizeDetermine = con.createStatement();
						ResultSet rsSizeDetermine = stSizeDetermine
								.executeQuery(SIZE_EXIST_QUERY);
						int val = 0;
						if (rsSizeDetermine.next()) {
							val = rsSizeDetermine.getInt(1);
						}

						rsSizeDetermine.close();
						stSizeDetermine.close();

						// Step 3.2 Determine size
						if (val > 0) {
							PreparedStatement ps = con
									.prepareStatement(DBSIZE_QUERY);
					
							ps.setLong(1, dboid);
							ResultSet rsSize = ps.executeQuery();
							if (rsSize.next()) {
								this.dbSize.add(rsSize.getString("dbsize"));
							}
							
							ps = con.prepareStatement(SERVER_UPTIME_QUERY);
							ResultSet rsServerStartTime = ps.executeQuery();
							if(rsServerStartTime.next()) {
								this.serverStartTime.add(rsServerStartTime.getTimestamp("modification").toString());
							}
						} else {
							this.dbSize.add("Size not available");
							this.serverStartTime.add("Not available");
						}

						count++;

					} catch (SQLException sqe) {
						sqe.printStackTrace();
					} finally {
						if (con != null) {
							try {
								con.close();
							} catch (SQLException e) {
							}
						}
					}
				}

			}
		}
	}

	/**
	 * Does encoding lookup
	 * 
	 * @param enc
	 *            The encoding integer
	 * @return String the encoding
	 */
	private String getEncoding(int enc) {
		String _encoding = null;
		switch (enc) {
			case 0 :
				_encoding = "SQL_ASCII";
				break;
			case 1 :
				_encoding = "EUC_JP";
				break;
			case 2 :
				_encoding = "EUC_CN";
				break;
			case 3 :
				_encoding = "EUC_KR";
				break;
			case 4 :
				_encoding = "EUC_TW";
				break;
			case 5 :
				_encoding = "JOHAB";
				break;
			case 6 :
				_encoding = "UNICODE";
				break;
			case 7 :
				_encoding = "MULE_INTERNAL";
				break;
			case 8 :
				_encoding = "LATIN1";
				break;
			case 9 :
				_encoding = "LATIN2";
				break;
			case 10 :
				_encoding = "LATIN3";
				break;
			case 11 :
				_encoding = "LATIN4";
				break;
			case 12 :
				_encoding = "LATIN5";
				break;
			case 13 :
				_encoding = "LATIN6";
				break;
			case 14 :
				_encoding = "LATIN7";
				break;
			case 15 :
				_encoding = "LATIN8";
				break;
			case 16 :
				_encoding = "LATIN9";
				break;
			case 17 :
				_encoding = "LATIN10";
				break;
			case 18 :
				_encoding = "WIN1256";
				break;
			case 19 :
				_encoding = "TCVN";
				break;
			case 20 :
				_encoding = "WIN874";
				break;
			case 21 :
				_encoding = "KO18";
				break;
			case 22 :
				_encoding = "WIN";
				break;
			case 23 :
				_encoding = "ALT";
				break;
			case 24 :
				_encoding = "ISO_8859_5";
				break;
			case 25 :
				_encoding = "ISO_8859_6";
				break;
			case 26 :
				_encoding = "ISO_8859_7";
				break;
			case 27 :
				_encoding = "ISO_8859_8";
				break;
			case 28 :
				_encoding = "WIN1250";
				break;
			case 29 :
				_encoding = "SJIS";
				break;
			case 30 :
				_encoding = "BIG5";
				break;
			case 31 :
				_encoding = "GBK";
				break;
			case 32 :
				_encoding = "UHC";
				break;
			case 33 :
				_encoding = "GB18030";
				break;
			default :
				_encoding = "Unknown (" + enc + ")";
				break;
		}
		return _encoding;
	}

	/**
	 * Returns true if we have more records
	 */
	public boolean next() throws JRException {
		if (count > 0) {
			count--;
			return true;
		}
		return false;
	}

	/**
	 * Gets value of a field
	 */
	public Object getFieldValue(JRField field) throws JRException {

		// determine field and return appropriate value

		if (field.getName().equals("source")) {
			return this.database_name.get(count);
		} else if (field.getName().equals("version")) {
			return this.version.get(count);
		} else if (field.getName().equals("owner")) {
			return this.owner.get(count);
		} else if (field.getName().equals("encoding")) {
			return this.encoding.get(count);
		} else if (field.getName().equals("numbackends")) {
			return this.numbackends.get(count);
		} else if (field.getName().equals("session_count")) {
			return this.session_count.get(count);
		} else if (field.getName().equals("xact_commit")) {
			return this.xact_commit.get(count);
		} else if (field.getName().equals("xact_rollback")) {
			return this.xact_rollback.get(count);
		} else if (field.getName().equals("blks_read")) {
			return this.blks_read.get(count);
		} else if (field.getName().equals("blks_hit")) {
			return this.blks_hit.get(count);
		} else if (field.getName().equals("dbsize")) {
			return this.dbSize.get(count);
		} else if (field.getName().equals("modification")) {
			return this.serverStartTime.get(count);
		} else if (field.getName().equals("qlogging")) {
                        return this.qlogging.get(count);
                } else if (field.getName().equals("show_image")) {
                        int i = 0 ;                                              
                        boolean sh_image = true;
                        for (i=0;i<this.stats_block_level.size();i++) {
                            if (this.stats_block_level.get(i).equals("on"))
                                sh_image = false;  
                        }
                        return new java.lang.Boolean(sh_image);
                }
                

		return null;
	}

}
