#!/bin/bash
################################################################################
# Autor     : Guilherme Augusto da Rocha Silva                                 #
# Data      : 20/02/2006                                                       #
# Objetivo  : Verificar o tamanho de uma ou mais bases de dados do PostgreSQL. #
# Observao: Para uso no Linux.                                               #
################################################################################

################################################################################
# Variveis.
################################################################################
script=$(basename ${0})              # Nome deste programa.
versao="0.1"                         # Verso deste programa.
release="20060706a"                  # Release deste programa.
versao_pg="8.1"                      # Verso corrente do PostgreSQL.
usuario=$(whoami);                   # Usurio corrente.

# Definio do diretrio da instncia ($PGDATA).
if [ ! -z ${PGDATA} ]; then
   dir_data=${PGDATA}                # Associao com a varivel de ambiente $PGDATA.
elif [ ! -z ${POSTGRES_DATA} ]; then
   dir_data=${POSTGRES_DATA}         # Associao com a varivel alternativa de ambiente $POSTGRES_DATA.
else
   dir_data="/var/lib/postgres/data" # Associao com o path default.
fi
dir_base=${dir_data}"/base"          # Diretrio das bases de dados.
dir_xlog=${dir_data}"/pg_xlog"       # Diretrio dos logs de transao.
dir_clog=${dir_data}"/pg_clog"       # Diretrio dos logs de controle de transao.

# Controle para exibio de informaes de debug (0 = no debugar; 1 = debugar).
if [ ! -z "$1" ]; then
   debugar=1
else
   debugar=0
fi

################################################################################
# Funes.
################################################################################
# Funo para abortar exibindo mensagem.
function abortar() {
   if [ ! -z "$1" ]; then linha=$1; else linha=${LINENO}; fi
   if [ ! -z "$2" ]; then sinal=$2; else sinal=1; fi
   if [ ! -z "$3" ]; then texto=$3; else texto="Abortando a execuo devido a um erro desconhecido!"; fi
   echo -e "\nERRO:\t"${texto}
   if [ ${debugar} -gt 0 ]; then echo -e "\n\t(DEBUG: linha="${linha}"; tempo="${SECONDS}")"; fi # gars - informao de debug.
   echo ""
   exit ${sinal}
}
# Funo para exibir cabecalho do programa.
function exibir_cabecalho() {
   msg="################################################################################"
   msg=${msg}"\n# ${script} (v${versao}, r${release})"
   msg=${msg}"\n# Servidor : $HOSTNAME"
   msg=${msg}"\n# Data/hora: $(date +%d\/%m\/%Y\ %H\:%M\:%S\ %Z)"
   msg=${msg}"\n################################################################################"
   echo -e ${msg}
}
# Funo para verificar a instncia do PostgreSQL (PGDATA).
function verifica_pgdata() {
   formato_pg=$(cat ${dir_data}"/PG_VERSION" 2> /dev/null)
   if [ ! -d ${dir_data} ] || [ -z "${formato_pg}" ]; then
      abortar ${LINENO} 1 "No existe um diretrio de trabalho vlido para PostgreSQL."
   else
      if [ "${formato_pg}" != "${versao_pg}" ]; then
         abortar ${LINENO} 1 "Verso estrutural da instncia difere da verso atual do PostgreSQL."
      fi
   fi
   if [ ! -d ${dir_data}/base ]; then
      abortar ${LINENO} 1 "No existe uma instncia do PostgreSQL em ${dir_data}."
   fi
}
# Funo para verificar se o servio do PostgreSQL est ativo.
function verifica_servico_postgresql() {
   postmaster_pid=${dir_data}"/postmaster.pid"
   if [ -r "${postmaster_pid}" ]; then
      processo_pg=$(COLUMNS=180 ps -p $(cat ${postmaster_pid} | head -1) | grep -v "PID" | awk '{print $NF}') # copiado do script ('/etc/init.d/postgresql').
      if [ "${processo_pg}" != "postmaster" ]; then
         abortar ${LINENO} "1" "O servio PostgreSQL est parado!"
      fi
   else
      abortar ${LINENO} 1 "O servio PostgreSQL est parado!"
   fi
}
# Funo para gerar arrays de oids e de nomes (ambos sincronizados entre si pelas respectivas posies de ndices).
# Retorna 0 e gera os arrays "id" e "bd".
function gerar_listas() {
   i=0 # ndice do array "bases".
   j=0 # ndice dos arrays "id" e "bd".
   sql="SELECT oid || ' ' || datname FROM pg_database WHERE datallowconn IS TRUE AND datname != 'template0' ORDER BY datname"
   # Criao de array monovalorado, composto por oid e nome da base, em linhas subsequentes (oid na 1a, nome na 2a, oid na 3a, nome na 4a...).
   bases=($(psql -U postgres template1 -t -c "${sql}"))
   while [ ${i} -lt ${#bases[*]} ]
   do
      modulo=$((${i} % 2)) # Definio se o ndice atual  par ou mpar.
      if [ ${modulo} -eq 0 ]; then
         ids[${j}]=${bases[${i}]} # Montagem de array "id".
         j=$((${j} + 1)) # Incremento do ndice de "id" e "bd".
      elif [ ${modulo} -eq 1 ]; then
         bds[$((${j} - 1))]=${bases[${i}]} # Montagem de array "bd".
      fi
      i=$((${i} + 1)) # Incremento do ndice de "bases".
   done
   return 0
}

################################################################################
# Verificaes iniciais.
################################################################################
# Exibio de ttulo.
exibir_cabecalho

if [ "${usuario}" != "root" ] && [ "${usuario}" != "postgres" ]; then
   abortar ${LINENO} 1 "Este programa deve ser executado apenas\n\tpor super-usurios do sistema operacional!"
fi
# Verifica se o diretrio 'data'  uma instncia de bancos de dados.
verifica_pgdata
# Verifica se o servio do PostgreSQL est ativo.
verifica_servico_postgresql

################################################################################
# Processamento.
################################################################################
# Gerao dos arrays de oids ("ids") e de nomes ("bds") das bases de dados.
gerar_listas

# Exibio dos tamanhos individuais das bases de dados.
echo -e "\n== Tamanho da(s) base(s) =="
k=0 # Contador dos arrays
while [ ${k} -lt ${#ids[*]} ]
do
   alvo=${dir_base}"/"${ids[$k]}"/"
   if [ -d ${alvo} ]; then
      tam=$(du -hsL ${alvo} | awk '{print $1}')
      echo -e "   ["${ids[$k]}"] "${bds[$k]}"\t--->\t"${tam}
   fi
   k=$((${k} + 1)) # atualizao da posio atual
done
# Exibio de tamanhos dos demais diretrios e total da instncia.
echo -e "
== Totalizaes ==
   Bases de dados ativas              : ${k}
   Tamanho das bases de dados         : $(du -hsL ${dir_base} | awk '{print $1}')
   Arquivos de logs de transao (WAL): $(ls -1 ${dir_xlog} | wc -l)
   Tamanho dos logs de transao (WAL): $(du -hsL ${dir_xlog} | awk '{print $1}')
   Tamanho total da instncia (PGDATA): $(du -hsL ${dir_data} | awk '{print $1}')

== Uso do disco =="
df -hT
echo ""
