#!/usr/local/bin/ruby

require "test/unit"
require "pqa.rb"

class QueryTest < Test::Unit::TestCase
	def test_add
		q = Query.new
		q.append("test")
		assert(q.to_s == " test", "append should insert leading space, instead:" + q.to_s + ":")	
	end
	def test_normalize_names
		assert(Query.new("WHERE unix_group_name='wtr'").normalize == "WHERE unix_group_name=''", "normalize should remove stuff in quotes")
	end
	def test_normalize_spaces
		assert(Query.new("WHERE    unix_group_name='wtr'").normalize == "WHERE unix_group_name=''", "normalize should squash spaces")
	end
	def test_normalize_numbers
		assert(Query.new("WHERE foo=213345").normalize == "WHERE foo=0", "normalize should normalize numbers")
	end
	def test_is_select
		assert(Query.new("SELECT * from users").is_select, "SELECT detection failed")
		assert(Query.new("select * from users").is_select, "SELECT detection failed")
	end
	def test_is_delete
		assert(Query.new("DELETE * from users"), "DELETE detection failed")
	end
	def test_is_insert
		assert(Query.new("insert into users values etc etc"), "INSERT detection failed")
	end
	def test_sort 
		#q = Query.new("foo")
		#q.duration = 2.0
		#q1 = Query.new("foo")
		#q.duration = 3.0
		#x = [q, q1]
		#x.sort {|a,b| a.duration <=> b.duration }
	end
end

class SyslogLineTest < Test::Unit::TestCase
	QUERY_LINE = SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755] LOG:  query: SELECT * FROM plugins")
	QUERY_LINE_WITH_DASH_IN_CONNECTION_ID = SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755-1] LOG:  query: SELECT * FROM plugins")
	RANDOM_SYSLOG_LINE = SyslogLine.new("Mar  9 14:29:05 hal modprobe: modprobe: Can't locate module sound-service-1-0")
	def test_is_pg_log_entry
		assert(QUERY_LINE.is_pg_log_entry, "Didn't recognize legit PG log entry")
		assert(!RANDOM_SYSLOG_LINE.is_pg_log_entry, "False positive")
	end
	def test_connection_id_no_dash
		assert(QUERY_LINE.connection_id == "2755", "Couldn't parse connection id: #{QUERY_LINE.connection_id}")
	end
	def test_connection_id_with_dash
		assert(QUERY_LINE_WITH_DASH_IN_CONNECTION_ID.connection_id == "2755", "Couldn't parse connection id: #{QUERY_LINE_WITH_DASH_IN_CONNECTION_ID.connection_id}")
	end
	def test_is_new_query	
		assert(QUERY_LINE.is_new_query, "Didn't identify new query")
	end
	def test_parse_query_all
		assert(QUERY_LINE.parse_query_segment == "SELECT * FROM plugins", "Didn't parse query segment correctly: #{QUERY_LINE.parse_query_segment}")
	end
	def test_parse_query_partial
		txt = SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2756-3] ^I^I^IFROM groups").parse_query_segment
		assert(txt ="FROM groups", "Didn't parse query segment correctly:" + txt + ":")
	end
end

class PostgresLogLineTest < Test::Unit::TestCase
	WITH_DATE = "2004-04-27 10:54:18 LOG:  query: SELECT language_code FROM supported_languages WHERE language_id='1'"
	WITH_PID = "[1234] LOG:  query: SELECT language_code FROM supported_languages WHERE language_id='1'"
	WITH_STATEMENT_PREAMBLE = "[1234] LOG:  statement: SELECT language_code FROM supported_languages WHERE language_id='1'"
	WITHOUT_DATE = "LOG:  query: SELECT language_code FROM supported_languages WHERE language_id='1'"
	DURATION = "LOG:  duration: 0.000496 sec"
	NON_QUERY = "postmaster successfully started"
	QUERY_CONTINUATION = "\t\tfrom users"
	def test_snip_date
		line = PostgresLogLine.new(WITH_DATE)
		assert(/^LOG/.match(line.text).size == 1, "Didn't snip date correctly")
		line = PostgresLogLine.new(WITHOUT_DATE)
		assert(/^LOG/.match(line.text).size == 1, "Failed to process line without date")
	end
	def test_snip_pid
		line = PostgresLogLine.new(WITH_PID)
		assert(/^LOG/.match(line.text).size == 1, "Didn't snip pid correctly")
	end
	def test_is_new_query
		line = PostgresLogLine.new(WITH_DATE)
		assert(line.is_new_query, "Failed to identify new query with 'query:' preamble")
		line = PostgresLogLine.new(WITH_STATEMENT_PREAMBLE)
		assert(line.is_new_query, "Failed to identify new query with 'statment:' preamble")
		line = PostgresLogLine.new(NON_QUERY)
		assert(!line.is_new_query, "Failed to identify non-query-line query")
	end
	def test_query_continuation
		line = PostgresLogLine.new(QUERY_CONTINUATION)
		assert(!line.is_new_query, "False positive on a query continuation")
		assert(line.is_continuation, "Failed to identify query continuation")
	end
	def test_parse_segment_from_new_query
		line = PostgresLogLine.new(WITHOUT_DATE)
		assert(line.parse_query_segment == "SELECT language_code FROM supported_languages WHERE language_id='1'", "Failed to parse query segment: #{line.parse_query_segment}")
	end
	def test_parse_segment_from_continuation
		line = PostgresLogLine.new(QUERY_CONTINUATION)
		assert(line.parse_query_segment == "from users", "Failed to parse query segment from continuation")
	end
	def test_parse_duration
		line = PostgresLogLine.new(DURATION)
		assert(line.is_duration_line, "Failed to recognize duration line")
		assert(line.parse_duration_segment == 0.000496, "Failed to correctly parse duration line: #{line.parse_duration_segment}")
	end
end

class AccumulatorTest < Test::Unit::TestCase
	def test_new_line
		a = Accumulator.new
		assert(a.queries.empty?, "Wrong initial query array size")
		a.new_query_start("select * from")
		a.query_continuation("users")
		a.new_query_start("select buz from bar")
		assert(a.queries.size == 1, "Failed to start new query when a new query line was hit")
	end
	def test_duration_ends_current_query
		a = Accumulator.new
		a.new_query_start("select * from")
		a.set_duration("3.14159")
		assert(a.queries.size == 1, "Failed to start new query when a new query line was hit")
	end	
	def test_partial_input
		a = Accumulator.new
		a.set_duration("3.14159")
		assert(a.queries.empty?, "Failed to discard duration when it's the first line")
		a = Accumulator.new
		a.query_continuation("from users")
		assert(a.queries.empty?, "Failed to discard query continuation when it's the first line")
	end
end
