/* $Id: type.c,v 1.22 2005/12/28 23:54:28 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres metatyping & caching
 */
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_cast.h>
#include <nodes/params.h>
#include <parser/parse_type.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/tupledesc.h>
#include <pypg/error.h>

#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/array.h>
#include <pypg/type/bitwise.h>
#include <pypg/type/record.h>
#include <pypg/type/network.h>
#include <pypg/type/numeric.h>
#include <pypg/type/geometric.h>
#include <pypg/type/textual.h>
#include <pypg/type/timewise.h>
#include <pypg/type/system.h>

char *PyPgType_SourceModKWL[] = {"source", "mod", NULL};
char *PyPgType_SourceKWL[] = {"source", NULL};
/*
 * Oid to PyPgType_Type mapping
 */
static PyObj cache;

int _PyPgType_Ready(PyPgTypeObject *, Oid);
static PyObj
type_new_from_oid(Oid typoid)
{
	PyObj key, rob;

	if (!OidIsValid(typoid))
	{
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError, "invalid Oid given for type lookup");
		return(NULL);
	}

	key = PyPg_oid_FromOid(typoid);
	if (key == NULL) return(NULL);
	rob = PyDict_GetItem(cache, key);
	if (rob == NULL && !PyErr_Occurred())
	{
		/*
		 * Note: Lookup the builtin *after* hitting the cache.
		 * This allows builtins to be overridden per desire.
		 */
		rob = PyPgType_LookupBuiltin(typoid);
		if (rob == NULL)
		{
			/* nothing in cache. no builtin either. time to create a subclass */
			HeapTuple ht;
			Form_pg_type ts;
			PyObj base, bases, nbd;
			ht = SearchSysCache(TYPEOID, ObjectIdGetDatum(typoid), 0, 0, 0);
			if (ht == NULL)
			{
				PyErr_Format(PyExc_LookupError, "failed to find type '%d'", typoid);
				Py_DECREF(key);
				return(NULL);
			}
			ts = TYPESTRUCT(ht);
			if (ts->typtype == 'p')
			{
				rob = Py_None;
				ReleaseSysCache(ht);
				PyDict_SetItem(cache, key, rob);
				Py_DECREF(key);
				if (PyErr_Occurred())
					return(NULL);
				else
					return(rob);
			}
			if (ts->typtype == 'c')
				base = (PyObj) &PyPg_record_Type;
			else if (ts->typelem != 0 && ts->typlen == -1)
				base = (PyObj) &PyPgArray_Type;
			else
				base = (PyObj) &PyPgArbitrary_Type;
			ReleaseSysCache(ht);
			bases = PyTuple_New(1);
			PyTuple_SET_ITEM(bases, 0, base);
			Py_INCREF(base);
			nbd = PyTuple_New(3);
			PyTuple_SET_ITEM(nbd, 0, PyString_FromString(NameStr(ts->typname)));
			PyTuple_SET_ITEM(nbd, 1, bases);
			PyTuple_SET_ITEM(nbd, 2, PyDict_New());
			rob = PyType_Type.tp_new((PyTypeObject *) base->ob_type, nbd, NULL);
			Py_DECREF(nbd);
			if (rob != NULL)
			{
				if (_PyPgType_Ready((PyPgTypeObject *) rob, typoid) < 0
				|| PyDict_SetItem(cache, key, rob) < 0)
				{
					Py_DECREF(rob);
					rob = NULL;
				}
				else if (base == (PyObj) &PyPgArray_Type)
				{
					PyObj elmtypob;
					elmtypob = type_new_from_oid(ts->typelem);
					PyPgType_FixElementType(rob, elmtypob);
					if (elmtypob == NULL)
					{
						Py_DECREF(rob);
						rob = NULL;
					}
				}
			}
		}
	}
	Py_DECREF(key);

	return(rob);
}

static PyObj
type_lookup(PyObj subtype, PyObj key)
{
	return(type_new_from_oid(Oid_FromPyObject(key)));
}

static PyObj
type_lookup_by_name(PyObj subtype, PyObj key)
{
	return(NULL);
}

static PyMethodDef PyPgType_Methods[] = {
	{"Lookup", (PyCFunction) type_lookup, METH_O | METH_CLASS,
	"lookup a Postgres type associated with the Oid"},
	{"LookupByName", (PyCFunction) type_lookup_by_name, METH_O | METH_CLASS,
	"lookup a Postgres type associated with the given name"},
	{NULL}
};

static PyObj
type_id(PyObj self, void *unused)
{
	return(PyPg_oid_FromOid(PyPgType_FetchOid(self)));
}

static PyGetSetDef type_getset[] = {
	{"typid", type_id},
	{NULL}
};

PyDoc_STRVAR(PyPgType_Type_Doc,
	"Python interface for the Postgres type"
);
PyTypeObject PyPgType_Type = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,										/* ob_size */
	"Postgres.Type",					/* tp_name */
	sizeof(PyPgHeapTypeObject),	/* tp_basicsize */
	0,										/* tp_itemsize */
	NULL,									/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	NULL,									/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	NULL,									/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,				/* tp_flags */
	PyPgType_Type_Doc,				/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgType_Methods,					/* tp_methods */
	NULL,									/* tp_members */
	type_getset,						/* tp_getset */
	&PyType_Type,						/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	NULL,									/* tp_new */
};

Oid
PyObject_FetchOid(PyObj tp)
{
	Oid typoid = InvalidOid;
	PyObj ob, exc = NULL, val, tb;
	Assert(PyPgType_Check(tp));

	if (PyErr_Occurred())
	{
		PyErr_Fetch(&exc, &val, &tb);
		PyErr_Clear();
	}

	ob = PyObject_GetAttrString(tp, "typid");
	if (ob == NULL) goto fail;

	typoid = (Oid) PyInt_AsLong(ob);
	Py_DECREF(ob);

	return(typoid);
fail:
	if (exc != NULL)
		PyErr_Restore(exc, val, tb);
	return(InvalidOid);
}

PyObj
PyPgType_FromOid(Oid to)
{
	return(type_new_from_oid(to));
}

char *
PgTypeName_FromOid(Oid typoid)
{
	char *pstr;
	HeapTuple ttup;

	ttup = SearchSysCache(TYPEOID, typoid, 0, 0, 0);
	pstr = pstrdup(NameStr(TYPESTRUCT(ttup)->typname));
	ReleaseSysCache(ttup);

	return(pstr);
}

Oid
TypeOidMod_FromPyObject(PyObj ob, int32 *typmod)
{
	Oid typoid = InvalidOid;
	*typmod = -1;

	if (PyString_Check(ob))
	{
		parseTypeString(PyString_AS_STRING(ob), &typoid, typmod);
	}
	else if (PyPgType_Check(ob))
	{
		typoid = PyPgObject_FetchOid(ob);
	}
	else
	{
		typoid = Oid_FromPyObject(ob);
	}

	return(typoid);
}

Oid
TypeOid_FromPyObject(PyObj ob)
{
	int32 mod;
	return(TypeOidMod_FromPyObject(ob, &mod));
}

PyObj
PyPgType_FromPyObject(PyObj ob)
{
	volatile Oid typoid = InvalidOid;
	volatile PyObj rob = NULL;

	/* XXX: need to look into getting this changed */
	PgError_TRAP(typoid = TypeOid_FromPyObject(ob));
	if (typoid == InvalidOid)
	{
		if (!PyErr_Occurred())
		{
			PyObj obstr;
			obstr = PyObject_Str(ob);
			PyErr_Format(PyExc_TypeError,
				"Postgres Type '%s' does not exist", PYASSTR(obstr));
			DECREF(obstr);
		}
	}
	else
		rob = type_new_from_oid(typoid);

	return(rob);
}

PyObj
PyPgType_LookupBuiltin(Oid typoid)
{
	PyObj rob;

	switch (typoid)
	{
#define ACTION(name, oid) \
		case oid: rob = ((PyObj) &PyPgTypeID(name)); break;
		PyPgType_Builtins()
#undef ACTION

		default:
			rob = NULL;
		break;
	}
	if (rob && !PyType_HasFeature((PyTypeObject *) rob, Py_TPFLAGS_READY))
	{
		if (PyPgType_Ready((PyPgTypeObject *) rob, typoid) < 0)
			return(NULL);
	}

	Py_XINCREF(rob);
	return(rob);
}

struct TypeCacheEntry AbstractTypeEntry = {
	InvalidOid, 0,
};

int
_PyPgType_Ready(PyPgTypeObject *subtype, Oid typoid)
{
	TypeCacheEntry *tc;

	if (OidIsValid(typoid))
	{
		HeapTuple ht = NULL;
		tc = lookup_type_cache(typoid, TYPECACHE_TUPDESC);
		PG_TRY();
		{
			HeapTuple typtup;
			Form_pg_type ts;
			FmgrInfo *flinfo;

			ht = SearchSysCache(TYPEOID, ObjectIdGetDatum(typoid), 0, 0, 0);
			typtup = ht;
			ts = TYPESTRUCT(typtup);

			flinfo = &PyPgType_FetchTypeInput(subtype);
			fmgr_info_cxt(ts->typinput, flinfo, PythonMemoryContext);
			Assert(flinfo->fn_retset == false);

			flinfo = &PyPgType_FetchTypeOutput(subtype);
			fmgr_info_cxt(ts->typoutput, flinfo, PythonMemoryContext);
			Assert(flinfo->fn_retset == false);

			flinfo = &PyPgType_FetchTypeReceive(subtype);
			fmgr_info_cxt(ts->typreceive, flinfo, PythonMemoryContext);
			Assert(flinfo->fn_retset == false);

			flinfo = &PyPgType_FetchTypeSend(subtype);
			fmgr_info_cxt(ts->typsend, flinfo, PythonMemoryContext);
			Assert(flinfo->fn_retset == false);

			ht = NULL;
			ReleaseSysCache(typtup);
		}
		PG_CATCH();
		{
			if (ht != NULL)
				ReleaseSysCache(ht);
		}
		PG_END_TRY();
	}
	else
	{
		tc = &AbstractTypeEntry;
	}

	PyPgType_FixTypeCache(subtype, tc);
	PyPgType_FixElementType(subtype, NULL);
	if (tc->tupDesc != NULL)
	{
		PyObj tdo;
		tdo = PyPgTupleDesc_New(tc->tupDesc);
		if (tdo == NULL) return(-1);
		PyPgType_FixPyPgTupleDesc(subtype, tdo);
	}

	/*
	 * Fix the array's typ_element.
	 */
	if (tc != &AbstractTypeEntry
		&& PyObject_IsSubclass((PyObj) subtype, (PyObj) &PyPgArray_Type))
	{
		bool isnull;
		Oid typelem = 0;
		HeapTuple typ_tuple = NULL;
		PyObj typ_element;
		typ_tuple = SearchSysCache(TYPEOID,
				ObjectIdGetDatum(typoid), 0, 0, 0);
		if (typ_tuple == NULL)
		{
			PyErr_SetString(PyExc_TypeError,
				"failed to lookup array's element type");
			return(-1);
		}

		typelem = DatumGetObjectId(
			SysCacheGetAttr(TYPEOID,
				typ_tuple, Anum_pg_type_typelem, &isnull)
		);
		ReleaseSysCache(typ_tuple);

		if (isnull || !OidIsValid(typelem))
		{
			PyErr_SetString(PyExc_TypeError,
				"invalid type element column for array");
			return(-1);
		}

		typ_element = type_new_from_oid(typelem);
		if (typ_element == NULL) return(-1);
		PyPgType_FixElementType(subtype, typ_element);
	}

	return(0);
}

int
PyPgType_Ready(PyPgTypeObject *subtype, Oid typoid)
{
	if (PyType_Ready((PyTypeObject *) subtype) < 0)
		return(-1);

	if (subtype->type.ob_type != &PyPgType_Type)
	{
		PyErr_Format(PyExc_TypeError,
			"invalid ob_type on type '%s', expecting PyPgType_Type, "
			"but got '%s' instead",
			subtype->type.tp_name,
			subtype->type.ob_type->tp_name);
		return(-1);
	}

	if (_PyPgType_Ready(subtype, typoid) < 0)
		return(-1);

	return(0);
}

int
PyPgType_Initialize(void)
{
	cache = PyDict_New();
	if (cache == NULL) return(-1);

	/*
	 * Ready the fundamental types and a few select builtins.
	 *
	 * Others will be readied upon use.
	 */
	if (PyType_Ready(&PyPgType_Type) < 0
		||	PyPgType_Ready(&PyPgObject_Type, 0) < 0
		||	PyPgType_Ready(&PyPgPseudo_Type, 0) < 0
		||	PyPgType_Ready(&PyPgArbitrary_Type, 0) < 0
		||	PyPgType_Ready(&PyPgArray_Type, 0) < 0
		|| PyPgType_Ready(&PyPgTypeID(record), RECORDOID)
		|| PyPgType_Ready(&PyPgTypeID(attribute), PG_ATTRIBUTE_RELTYPE_OID)
		|| PyPgType_Ready(&PyPgTypeID(bool), BOOLOID) < 0
		|| PyPgType_Ready(&PyPgTypeID(oid), OIDOID) < 0
		|| PyPgType_Ready(&PyPgTypeID(cid), CIDOID) < 0
		|| PyPgType_Ready(&PyPgTypeID(xid), XIDOID) < 0
		|| PyPgType_Ready(&PyPgTypeID(cstring), CSTRINGOID) < 0
		|| PyPgType_Ready(&PyPgTypeID(float8), FLOAT8OID) < 0
		|| PyPgType_Ready(&PyPgTypeID(point), POINTOID) < 0
	)
		return(-1);

	return(0);
}

/*
 * vim: ts=3:sw=3:noet:
 */
