/*
     Chronos Temporal Extensions
     Author:     Scott Bailey
     License:    BSD

    Copyright (c) 2009, Scott Bailey

    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are met:
        * Redistributions of source code must retain the above copyright
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright
          notice, this list of conditions and the following disclaimer in the
          documentation and/or other materials provided with the distribution.
        * Neither the name of the <organization> nor the
          names of its contributors may be used to endorse or promote products
          derived from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY Scott Bailey ''AS IS'' AND ANY
    EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
    DISCLAIMED. IN NO EVENT SHALL Scott Bailey BE LIABLE FOR ANY
    DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
    LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
    ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/*****************************************************************
*                     Additional Constructors
*****************************************************************/

-- [start, start + interval)
-- [start + (-interval), start)
CREATE OR REPLACE FUNCTION period(
	timestampTz,
	interval
) RETURNS period AS
$$
	SELECT CASE WHEN $2 >= INTERVAL '0 seconds'
	THEN period($1, $1 + $2)
	ELSE period($1 + $2, $1) END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;

-- [ts, ts + n seconds) or
-- [ts + (-n seconds), ts)
CREATE OR REPLACE FUNCTION period(
	timestampTz,
	numeric
) RETURNS period AS
$$
	SELECT CASE WHEN $2 >= 0 THEN
	period($1, $1 + ($2 * period_granularity()))
	ELSE period($1 + ($2 * period_granularity()), $1) END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;

-- period containing entire day
CREATE OR REPLACE FUNCTION period(date)
RETURNS period AS
$$
    SELECT period($1::timestampTz, ($1 + 1)::timestampTz);
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


/*****************************************************************
*                          Fact Functions
*****************************************************************/

-- Returns the length of p1 in seconds
CREATE OR REPLACE FUNCTION seconds(p1 period)
RETURNS numeric AS
$$
    SELECT EXTRACT(EPOCH FROM
      (next($1) - first($1)))::numeric;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION seconds(period)
IS 'Length of p1 in seconds';

--  Mid point of the period, I couldn't help myself on the name.
CREATE OR REPLACE FUNCTION mean_time(p1 period)
RETURNS timestampTz AS
$$
    SELECT first($1) + length($1) / 2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION mean_time(period)
IS 'Midpoint of period';

-- p1.first = p2.first and p1.last < p2.last
CREATE OR REPLACE FUNCTION starts(p1 period, p2 period)
RETURNS boolean AS
$$
    SELECT first($1) = first($2) AND next($1) < next($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION starts(period, period)
IS 'True if p1 has the same start time as p2 and ends before p2';


-- p1.last = p2.last and p1.first > p2.first
CREATE OR REPLACE FUNCTION ends(p1 period, p2 period)
RETURNS boolean AS
$$
    SELECT next($1) = next($2) AND first($1) > first($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION ends(period, period)
IS 'True if p1 has the same end time as p2 and starts after p2.';


/*****************************************************************
*                         Set Functions
*****************************************************************/

-- Returns the length (interval) of the overlap between p1 and p2
CREATE OR REPLACE FUNCTION overlap_length(p1 period, p2 period)
RETURNS interval AS
$$
    SELECT CASE WHEN overlaps($1, $2)
    THEN LEAST(next($1), next($2)) -
      GREATEST(first($1), first($2)) END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION overlap_length(period, period) 
IS 'Returns interval length of overlap between p1 and p2.
If the periods do not overlap NULL is returned.';


-- Like period_union except that periods do not
-- need to overlap
CREATE OR REPLACE FUNCTION range(p1 period, p2 period)
RETURNS period AS
$$
    SELECT period(LEAST(first($1), first($2)),
    GREATEST(next($1), next($2)));
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION range(period, period) 
IS 'Returns the smallest period containing all values in p1 and p2.
Like period_union except that periods do not have to be adjacent or overlap';

-- remove any values from p1 that are contained in p2
-- If you know p2 is NOT contained you can use period_minus
-- which returns a single period instead of a period array.
CREATE OR REPLACE FUNCTION period_exclude(p1 period, p2 period)
RETURNS period[] AS
$$ 
  SELECT array_agg(p)
  FROM (
    SELECT CASE WHEN first($1) < first($2)
    THEN period(first($1),
    LEAST(next($1), first($2))) END AS p

    UNION ALL
    
    SELECT CASE WHEN next($1) > next($2)
    THEN period(GREATEST(first($1), next($2)),
    next($1))END
  ) sub
  WHERE p IS NOT NULL;
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION period_exclude(p1 period, p2 period)
IS 'Remove all values of p1 that are in p2. Returns array 
because if p1 contains p2, p1 will be split into two periods.';


-- Enumerate all granules of size interval in period
CREATE OR REPLACE FUNCTION enumerate(p1 period, i interval)
RETURNS SETOF timestampTz AS
$$
  SELECT first($1) + ($2 * i)
  FROM generate_series(0,
    floor(seconds($1)/seconds($2))::int - 1) i
  WHERE $2 > INTERVAL '0 seconds';
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION enumerate(period, interval) 
IS 'Returns a set of timestamptz contained in p1 incremented by interval i';

-- Enumerate all seconds in period
CREATE OR REPLACE FUNCTION enumerate(p1 period)
RETURNS SETOF timestampTz AS
$$
  SELECT enumerate($1, period_granularity());
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION enumerate(period)
IS 'Returns a set of timestamptz contained in p1 incremented by period granule';


/*****************************************************************
*                       period functions
*****************************************************************/

-- pretty print
CREATE OR REPLACE FUNCTION to_char(p1 period)
RETURNS text AS
$$
    SELECT '[' || TO_CHAR(first($1), 'YYYY-MM-DD HH24:MI:SS') || 
    ', '  || TO_CHAR(next($1), 'YYYY-MM-DD HH24:MI:SS') || ')';
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION to_char(period)
IS 'Pretty print for period and cast function to text';

-- shifts entire period by interval
CREATE OR REPLACE FUNCTION period_shift(p1 period, i interval)
RETURNS period AS
$$
    SELECT period(first($1) + $2, next($1) + $2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_shift(period, interval)
IS 'Shift an entire period p1 by interval i';

-- Shifts a period array by interval
CREATE OR REPLACE FUNCTION period_shift(pa1 period[], i interval)
RETURNS period[] AS
$$
    SELECT array_agg(period(first(p) + $2, next(p) + $2))
    FROM unnest($1) p
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION period_shift(period[], interval)
IS 'Shift a period array by interval';

-- shifts entire period by n units(seconds)
CREATE OR REPLACE FUNCTION period_shift(p1 period, n numeric)
RETURNS period AS
$$
    SELECT period(first($1) + intv, next($1) + intv)
    FROM (
       SELECT period_granularity() * $2 AS intv
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION period_shift(period, numeric)
IS 'Shift an entire period p1 by n granules';

-- Shifts a period array by n units(seconds)
CREATE OR REPLACE FUNCTION period_shift(pa1 period[], n numeric)
RETURNS period[] AS
$$
    SELECT period_shift($1, period_granularity() * $2);
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION period_shift(period[], numeric)
IS 'Shift a period array by n granules';

-- add interval from duration of period
-- if interval is negative and longer than length(p1) then
-- period will have a 0 length
CREATE OR REPLACE FUNCTION period_grow(p1 period, i interval)
RETURNS period AS
$$
    SELECT CASE WHEN $2 > INTERVAL '0 second'
      THEN period(first($1), next($1) + $2)
    WHEN length($1) + $2 > INTERVAL '0 second'
      THEN period(first($1), next($1) + $2)
    ELSE period(first($1), first($1)) END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_grow(period, interval)
IS 'Extend the length of p1 by interval i';

-- add n seconds from the duration of period
CREATE OR REPLACE FUNCTION period_grow(p1 period, n numeric)
RETURNS period AS
$$
    SELECT CASE WHEN intv > zero
      THEN period(first($1), next($1) + intv)
    WHEN length($1) + intv > zero
      THEN period(first($1), next($1) + intv)
    ELSE period(first($1), first($1)) END
    FROM (
        SELECT INTERVAL '0 second' AS zero,
        period_granularity() * $2 AS intv
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_grow(period, numeric)
IS 'Extend the length of p1 by n period granules';

-- subtract interval from the duration of a period
CREATE OR REPLACE FUNCTION period_shrink(p1 period, i interval)
RETURNS period AS
$$    
    SELECT period_grow($1, -$2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_shrink(period, interval)
IS 'Reduce the length of period p1 by interval i';


-- subtract n seconds from the duration of a period
CREATE OR REPLACE FUNCTION period_shrink(period, numeric)
RETURNS period AS
$$
    SELECT period_grow($1, -$2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;
COMMENT ON FUNCTION period_shrink(period, numeric)
IS 'Reduce the length of period p1 by n period granules';


CREATE OR REPLACE FUNCTION period_measure(
  p1      IN   period,
  ctx     IN   period,
  x1      OUT  numeric,
  x2      OUT  numeric
) AS
$$
    SELECT (seconds(first(subject) - first(ctx))
        / ctx_duration * 100)::numeric,
    (seconds(next(subject) - first(ctx))
        / ctx_duration * 100)::numeric
    FROM (
        SELECT period_intersect($1, $2) AS subject,
        $2 AS ctx,
        seconds($2) AS ctx_duration
    ) sub
    WHERE seconds(subject) > 0 AND ctx_duration > 0;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_measure(period, period)
IS 'Measures the start and end times of p1 as percentages of period ctx.';

  
/*****************************************************************
*                             cast
*****************************************************************/

CREATE CAST (period AS text)
WITH FUNCTION to_char(period);

/*****************************************************************
*                       period operators
*****************************************************************/

CREATE OPERATOR ><(
  PROCEDURE = adjacent,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR &(
  PROCEDURE = period_intersect,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR +(
  PROCEDURE = period_grow,
  LEFTARG = period,
  RIGHTARG = interval
);

CREATE OPERATOR -(
  PROCEDURE = period_shrink,
  LEFTARG = period,
  RIGHTARG = interval
);

CREATE OPERATOR +(
  PROCEDURE = period_grow,
  LEFTARG = period,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = period_shrink,
  LEFTARG = period,
  RIGHTARG = numeric
);

/***************************************************************************
*                      array boolean functions
***************************************************************************/

CREATE OR REPLACE FUNCTION contains(
    pa period[],
    p  period
) RETURNS boolean AS
$$
    SELECT MAX(
       CASE WHEN contains(p, $2) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($1) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION contains(period[], period)
IS 'True if period p is contained in any period in pa';

CREATE OR REPLACE FUNCTION contains(
    pa period[],
    ts timestampTz
) RETURNS boolean AS
$$
    SELECT MAX(
      CASE WHEN contains(p, $2) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($1) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION contains(period[], timestampTz)
IS 'True if timestampTz ts contained in any period in pa';


CREATE OR REPLACE FUNCTION contained_by(
    p   period,
    pa  period[]
) RETURNS boolean AS
$$
    SELECT MAX(
       CASE WHEN contained_by($1, p) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($2) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION contained_by(period, period[])
IS 'True if period p is contained in any period in pa';


CREATE OR REPLACE FUNCTION contained_by(
    ts  timestampTz,
    pa  period[]
) RETURNS boolean AS
$$
    SELECT MAX(
      CASE WHEN contained_by($1, p) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($2) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION contained_by(timestampTz, period)
IS 'True if timestampTz is contained in any period in pa';

/***************************************************************************
*                        period array functions
***************************************************************************/

CREATE OR REPLACE FUNCTION range(pa period[])
RETURNS period AS
$$
    SELECT period(MIN(first(p)), MAX(next(p)))
    FROM unnest($1) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION range(period[])
IS 'Returns the smallest period that contains all of the values in array pa';


-- Convert period array to set of non-continuous periods
CREATE OR REPLACE FUNCTION reduce(pa period[])
RETURNS period[] AS
$$
    SELECT array_agg(t)
    FROM (
        SELECT period(start_time, MIN(end_time)) AS t
        FROM (
            SELECT DISTINCT first(p) AS start_time
            FROM unnest($1) p
            WHERE NOT contains($1, prior(p))
        ) AS t_in
        JOIN (
            SELECT next(p) AS end_time
            FROM unnest($1) p
            WHERE NOT contains($1, next(p))
        ) AS t_out ON t_in.start_time < t_out.end_time
        GROUP BY t_in.start_time
        ORDER BY t_in.start_time
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION reduce(period[])
IS 'Join overlapping and adjacent periods';

-- Set union function on period arrays
CREATE OR REPLACE FUNCTION period_union(
   pa1  IN period[],
   pa2  IN period[]
) RETURNS period[] AS
$$
   SELECT reduce($1 || $2);
$$ LANGUAGE 'sql' IMMUTABLE;
COMMENT ON FUNCTION period_union(period[], period[])
IS 'Return period array containing all values that are either in pa1 or pa2';

-- Set intersection operation on period arrays
CREATE OR REPLACE FUNCTION period_intersect(
    pa1 IN period[], 
    pa2 IN period[]
) RETURNS period[] AS
$$
    SELECT array_agg(t)
    FROM (
        SELECT period(start_time, MIN(end_time)) AS t
        FROM (
            SELECT DISTINCT first(p) AS start_time
            FROM unnest($1) p
            WHERE NOT contains($1, prior(p))
              AND contains($2, first(p))

            UNION

            SELECT DISTINCT first(p) 
            FROM unnest($2) p
            WHERE NOT contains($2, prior(p))
              AND contains($1, first(p))
        ) AS t_in
        JOIN (
            SELECT next(p) AS end_time
            FROM unnest($1) p
            WHERE NOT contains($1, next(p))
              AND contains($2, last(p))

            UNION ALL

            SELECT next(p)
            FROM unnest($2) p
            WHERE NOT contains($2, next(p))
              AND contains($1, last(p))
        ) AS t_out ON t_in.start_time < t_out.end_time
        GROUP BY t_in.start_time
        ORDER BY t_in.start_time
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_intersect(period[], period[])
IS 'Return period[] containing all values in both pa1 and pa2';

-- Set minus operation on period arrays
CREATE OR REPLACE FUNCTION period_exclude(
   pa1  IN period[], 
   pa2  IN period[]
) RETURNS period[] AS
$$
    SELECT array_agg(prd)
    FROM (
        SELECT period((t_in).start_time,
            MIN((t_out).end_time)) AS prd
        FROM (
            SELECT DISTINCT first(p) AS start_time
            FROM unnest($1) p
            WHERE NOT contains($2, first(p))
            AND NOT contains($1, prior(p))

            UNION

            SELECT DISTINCT next(p)
            FROM unnest($2) p
            WHERE contains($1, next(p))
            AND NOT contains($2, next(p))
        ) t_in
        JOIN (
            SELECT next(p) AS end_time
            FROM unnest($1) p
            WHERE NOT contains($1, next(p))

            UNION ALL

            SELECT first(p)
            FROM unnest($2) p
            WHERE contains($1, first(p))
              AND NOT contains($2, prior(p))
        ) t_out ON t_in.start_time < t_out.end_time
        GROUP BY (t_in).start_time
        ORDER BY (t_in).start_time
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;
COMMENT ON FUNCTION period_exclude(period[], period[])
IS 'Return period[] containing all values in pa1 that are not filtered out by pa2';


/*****************************************************************
*                      array boolean operators
*****************************************************************/

CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period[],
  RIGHTARG = period
);

CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period[],
  RIGHTARG = timestampTz
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = period,
  RIGHTARG = period[]
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = timestampTz,
  RIGHTARG = period[]
);

DROP OPERATOR IF EXISTS -(period, period);

CREATE OPERATOR -(
  PROCEDURE = period_exclude,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR -(
  PROCEDURE = period_exclude,
  LEFTARG = period[],
  RIGHTARG = period[]
);

CREATE OPERATOR &(
  PROCEDURE = period_intersect,
  LEFTARG = period[],
  RIGHTARG = period[]
);

CREATE OPERATOR +(
  PROCEDURE = period_union,
  LEFTARG = period[],
  RIGHTARG = period[]
);
